(function(){
	var duration = angular.module("durationPicker", []);

	duration.directive('durationPicker', function(){
		return {
			restrict: 'E',
			templateUrl: 'duration_index.html',
			scope:{'options': '='},
			controller: ['$scope', '$element','$timeout', function($scope, $element, $timeout){
				//DurationPicker controller api accessible 
				// in child directives and from onRegisterApi
				var dPCtrl = this;
				
				//Helper function to verify declared options
				function wasDeclared(configEntry){
					return typeof configEntry !== 'undefined';
				}
				
				//Initialise config option if it was declared or set a default value
				function initConfig(configEntry, defaultValue){
					return wasDeclared(configEntry)? configEntry: defaultValue;
				}
				
				//Option defaults
				
				//The pickerEndDate is the latest date that the picker displays 
				//defaut current moment(), this date is the most recent date that a picker can show.
				//To set the displayed date use api.setDateRange(beginDate, endDate)
				$scope.options.pickerEndDate = initConfig($scope.options.pickerEndDate, moment());
				
				//Debounce value
				$scope.options.debounce = initConfig($scope.options.debounce, 0);
				$scope.options.unit = initConfig($scope.options.unit, moment.duration(4, 'months'));
				$scope.options.group.groupCount = initConfig($scope.options.group.groupCount, 5);
				
				$scope.units = createUnits($scope.options.pickerEndDate, getLow($scope.options.pickerEndDate), $scope.options.group.groupCount, true);
				
				//Variable containing all dom elements 
				//related to the durationpicker (guide, highPicker, duration, lowPicker)
				dPCtrl.elements = {};
				
				dPCtrl.sliderHeights = {};
				
				//Calculates newHeight based on parameters and mouseevents
				dPCtrl.getNewHeight = function (e, initialHeight, mouseStart, isHigh){
					var delta;
					if(isHigh){
						delta = mouseStart.pageY - e.pageY;
					}else{
						delta = e.pageY - mouseStart.pageY;
					}
					return initialHeight + delta;
				};
				
				
				dPCtrl.getPosition = function (e){
					return {pageY: e.pageY, pageX: e.pageX};
				}
				
				//Creates units and set value limits
				// - a unit is the smallest mesure of a duration that the picker displays.
				function createUnits(high, low, count, isFirst){
					if(count > 0){
						var from = moment(low);
						var i = 0;var unit;var groupTo;
						var state;
						var res = [];
						
						while((unit = getUnit(from, high)).to.isBefore(high) && i < 400){
							from = moment(unit.to).subtract(1, 'seconds');
							i = i + 1;
							res.push(unit);
						};
						
						//Makes sure the ceilling is taken into account
						if(isFirst){
							unit.to = high;
							res.push(unit);
							groupTo = high;
						}else{
							groupTo = high;
						}
						res[0].type = 'group';
						res[0].name = $scope.options.group.key(high,low);
						res.reverse();
						
						/*res.push({from: low, to: groupTo, type :'group', 
							name: $scope.options.group.key(high,low)});*/
						var nextHigh = moment(low).subtract(1,"seconds");
						return res.concat(createUnits(nextHigh, getLow(nextHigh), count -1, false));
					}else{
						return [];
					}
				}
				
				//Creates a unit with default values
				function getUnit(start, end){
					var to = moment(start).add($scope.options.unit);
					return {from: start, to: to, type:'unit', c: 'pointBlank'};
				}
				
				//Returns a group low limit 
				//e.g if a moment(2017-07-19) is passed as the high parameter
				// and the roundTo parameter is year the function returns moment(2017-01-01) 
				function getLow(high){
					if(wasDeclared($scope.options.group) && wasDeclared($scope.options.group.roundTo)){
						var groupStart = moment(high).startOf($scope.options.group.roundTo);
						return groupStart;
					}
					return high;
				}
				
				//Called when all dom elements have been initialised
				//All onload processes are executed at this point
				$timeout(function () {
					dPCtrl.sliderHeights.guide = $(dPCtrl.elements.guide).height() ;
					dPCtrl.sliderHeights.picker = $(dPCtrl.elements.highPicker).height();
					dPCtrl.sliderHeights.durationMax = (dPCtrl.sliderHeights.guide-(dPCtrl.sliderHeights.picker*2));
					dPCtrl.sliderHeights.unit =Math.ceil(dPCtrl.sliderHeights.durationMax/$scope.units.length);
					
					//dPCtrl.setDateRange(moment().subtract(1, 'year'), moment());
					dPCtrl.elements.duration.updateIndicators();
					if($scope.options.onRegisterApi){
						$scope.options.onRegisterApi(dPCtrl);
					}
				});
				
				//Updates picker active sliders
				function updateActives(){
					$.each($scope.units, function( index, value ) {
						var start = $(value.domObj).position().top + parseInt( $(value.domObj).css("marginTop") );
						var end = start + $(value.domObj).height();
						if(start >= $scope.pxRangeStart && end <= $scope.pxRangeEnd){
							value.domObj.addClass("active");
						}else{
							value.domObj.removeClass("active");
						}
					});
				}
					
				//Sets duration to picker 
				//start parameter is a moment() object and 
				// end parameter is a moment() object.
				dPCtrl.setDateRange = function(start, end){
					var startPx = null;
					var endPx = null;
					var startPxValue;
					var endPxValue;
					$.each($scope.units, function( index, value ) {
						var includeStart = moment(start).isSame(value.from, 'day') || moment(start).isBefore(value.from, 'day') ;
						var includeEnd = moment(end).isSame(value.to, 'day') || moment(end).isAfter(value.to, 'day') ;
						if(includeStart &&includeEnd ){
							value.domObj.addClass("active");
									
							startPxValue = $(value.domObj).position().top + parseInt( $(value.domObj).css("marginTop") );
							endPxValue = startPxValue + $(value.domObj).height();
									
							if(startPx === null || startPx > startPxValue){
								startPx = startPxValue;
							}
							if(endPx === null || endPx < endPxValue){
								endPx = endPxValue;
							}
						}else{
							value.domObj.removeClass("active");
						}
					});
					$scope.pxRangeStart = startPx - Math.floor(dPCtrl.sliderHeights.unit);
					$scope.pxRangeEnd = endPx;
					var rangeStartPos = $scope.pxRangeStart < 0? 0:$scope.pxRangeStart;
					var rangeLength = endPx - rangeStartPos;
					
					dPCtrl.elements.duration.setDurationPosition(rangeStartPos, rangeLength);
				}
				
				var debounceTimeOut = null;
				
				//Called when the duration has changed (mouse up)
				dPCtrl.durationChanged = function(){
					dPCtrl.onModificationLeave();
					if($scope.options.debounce === 0 ){
						updateDuration();
					}else{
						debounceTimeOut = $timeout(updateDuration, $scope.options.debounce);
					}
				}
				
				//Private function to update the duration of the picker. 
				function updateDuration(){
					debounceTimeOut = null;
					dPCtrl.currentDuration = getMaxMin();
					$scope.options.onDurationChanged(dPCtrl.currentDuration.start, dPCtrl.currentDuration.end);
				}
				
				dPCtrl.currentDuration = {start: moment().subtract(1, 'year'), end: moment()}
				//Returns the max and min duration of the picker
				function getMaxMin(){
					var maxMin = {start: null, end: null};
					
					$.each($scope.units, function( index, value ) {
						//Exclude group units that contain multiple units.
						if(value.domObj.hasClass("active") ){
							if(maxMin.start === null || value.from.isBefore(maxMin.start)){
								maxMin.start = value.from;
							}
							if(maxMin.end === null || value.to.isAfter(maxMin.end)){
								maxMin.end = value.to;
							}
						}
						
					});
					return maxMin;
				}
				//return the height of one unit
				function findUnitHeight (){
					var i = 0;
					while($scope.units[i].type === "group" && i < $scope.units.length)i++;
					return $($scope.units[i].domObj).outerHeight(true);
				}
				
				//Called on mouse init event
				dPCtrl.dragInit = function(moveElement,destroy){
					dPCtrl.onModificationEnter();
					 // Store the object of the element which needs to be moved
					document.onmousemove = moveElement;
					document.onmouseup = destroy;
				    $("body")[0].style.userSelect = 'none';
				}
				
				//Called on mouse move event
				dPCtrl.move = function(){
					
				}
				
				//Called on mouse up event
				dPCtrl.destroy = function(){
					document.onmousemove = null;
				    document.onmouseup = null;
				    $("body")[0].style.userSelect = '';
				    dPCtrl.durationChanged();
				}
				
				dPCtrl.setPxRange = function(start, end){
					$scope.pxRangeStart = start;
					$scope.pxRangeEnd = end;
					updateActives();
				}
				
				//Called when a user starts duration modification (mouse down)
				dPCtrl.onModificationEnter = function(){
					if(debounceTimeOut !== null){
						$timeout.cancel(debounceTimeOut);
					}
					if($scope.options.onModificationEnter){
						$scope.options.onModificationEnter();
					}
				}
				
				//Called when a user leaves duration modification (mouse up) 
				//but before durationChange is called.  
				dPCtrl.onModificationLeave = function(){
					if($scope.options.onModificationLeave){
						$scope.options.onModificationLeave();
					}
				}
				
			}]
		}
	});
	
	//Directive for a duration unit. 
	duration.directive('durationUnit', function(){
		return {
			restrict: 'A',
			require: '^^durationPicker',
			link: function(scope, element, attrs, durationPickerCtrl){
				element.addClass(scope.unit.type);
				scope.unit.domObj = element;
				scope.pickDuration = function(){
					// console.log("My range :: Start ::" + scope.unit.from.format("ddd MMM DD YYYY") + " - Finish::" + scope.unit.to.format("ddd MMM DD YYYY"));
				}
			}
		}
	});
	
	duration.directive('durationSlider', function(){
		return {
			restrict: 'A',
			scope: true,
			require: '^^durationPicker',
			link: function(scope, element, attrs, durationAPICtrl){
				durationAPICtrl.elements.guide = element;
			}
		}
	});
	
	duration.directive('durationScrollbar', function(){
		return {
			restrict: 'A',
			require: '^^durationPicker',
			scope: true,
			link: function(scope, element, attrs, durationAPICtrl){
				durationAPICtrl.elements.durationScrollbar = element;
			}
		}
	});
	
	duration.directive('highPicker', function(){
		return {
			restrict: 'A',
			require: '^^durationPicker',
			scope: true,
			link: function(scope, element, attrs, durationAPICtrl){
				durationAPICtrl.elements.highPicker = element;
				

				var selected = null,
			    startPos = {}, y_elem = 0,total = 0,
				mouseStart = {}, initialHeight = 0;
				
				function _drag_init(e) {
					durationAPICtrl.dragInit(_move_elem, _destroy);
					
				    selected = durationAPICtrl.elements.durationScrollbar[0];
				    initialHeight = durationAPICtrl.elements.duration.height();
				    startPos = $(selected).position();
				    mouseStart = durationAPICtrl.getPosition(e);
				    total= $(durationAPICtrl.elements.guide).height();
				}

				function _move_elem(e) {
					durationAPICtrl.move();
					
				    var p = $(selected).position();
				    var newHeight = durationAPICtrl.getNewHeight(e, initialHeight, mouseStart, true);
				    var d1 =(newHeight - initialHeight) ;
				    var newPos = startPos.top - d1;
				    if (selected !== null){
				    	if(newPos < 0){
				    		newHeight = initialHeight + startPos.top;
				    		newPos = 0;
				    	}
				    	if(newHeight < 0){
				    		newHeight = 0 ;
				    		newPos = startPos.top + initialHeight;
				    	}
			    		newHeight = newHeight < durationAPICtrl.sliderHeights.unit ? durationAPICtrl.sliderHeights.unit : newHeight;
			    		durationAPICtrl.elements.duration.setHeight(newHeight);
			    		selected.style.top = newPos + 'px';
				    }
				}
				
				function _destroy() {
				    selected = null;
				    mouseStart = 0;
				    durationAPICtrl.destroy();
				}
				
				element[0].onmousedown = _drag_init;
			}
		}
	});
	
	//actualDuration directive for the actual duration.
	duration.directive('duration', ['$timeout', function($timeout){
		return {
			require: '^^durationPicker',
			restrict: 'A',
			scope: true,
			link: function(scope, element, attrs, durationAPICtrl){
				durationAPICtrl.elements.duration = {
						setHeight: function(newH){
							var v = $(durationAPICtrl.elements.durationScrollbar[0]).position().top;
							var start = v + $(element).position().top ;
							durationAPICtrl.setPxRange(start, newH + start);
							element.css('height', newH + 'px');
						},
						updatePosition: function(newPos){
							durationAPICtrl.elements.duration.updateIndicators();
							selected.style.top = newPos + 'px';
						},
						setDurationPosition: function(newStart, newH){
							// console.log("Height"+newH);
							element.css('height', newH + 'px');
							// console.log("Starting point"+newStart);
							durationAPICtrl.elements.durationScrollbar[0].style.top = newStart+'px';
						},
						updateIndicators: function(){
							var v = $(durationAPICtrl.elements.durationScrollbar[0]).position().top;
							var start = v + $(element).position().top ;
							var newH = $(element).height();
							durationAPICtrl.setPxRange(start, newH + start);
						},
						height: function(){
							return $(element).height();
						},
						position: function(){
							return $(element).position();
						}
						
				};
				
				var selected = null,max = 0,
				mouseStart = 0;
				
				function _drag_init(e) {
					durationAPICtrl.dragInit(_move_elem, _destroy);
					
				    selected = durationAPICtrl.elements.durationScrollbar[0];
				    startPos = $(selected).position().top;
				    mouseStart = durationAPICtrl.getPosition(e);
				    max = $(durationAPICtrl.elements.guide).height() - $(selected).height();
				}
				
				function _move_elem(e) {
					durationAPICtrl.move();
					
				    var newPos = startPos - (mouseStart.pageY - e.pageY);
				    if(newPos < 0){
				    	newPos = 0;
				    }else if(newPos  > max){
				    	newPos = max;
				    }
				    if (selected !== null) {
				    	durationAPICtrl.elements.duration.updatePosition(newPos);
				    }
				}
				
				function _destroy() {
				    selected = null;
				    mouseStart = 0;
				    durationAPICtrl.destroy();
				}
				
				element[0].onmousedown = _drag_init;
			}
		}
	}]);
	
	duration.directive('lowPicker', function(){
		return {
			restrict: 'A',
			require: '^^durationPicker',
			scope: true,
			link: function(scope, element, attrs, durationAPICtrl){
				durationAPICtrl.lowPicker = element;
				
				var selected = null,
				startPos = {}, y_elem = 0, total = 0,max = 0,
				mouseStart = {}, initialHeight = 0;
				
				
				function _drag_init(e) {
					durationAPICtrl.dragInit(_move_elem, _destroy);
				   
					
				    selected = durationAPICtrl.elements.durationScrollbar[0];
				    initialHeight = durationAPICtrl.elements.duration.height();
				    startPos = $(selected).height() + $(selected).position().top;
				    mouseStart = durationAPICtrl.getPosition(e);
				    total = $(durationAPICtrl.elements.guide).height();
				    max = initialHeight + (total - startPos);
				}
				
				function _move_elem(e) {
					durationAPICtrl.move();
					
				    var bottom = $(selected).height() + $(selected).position().top;
				    var newHeight = durationAPICtrl.getNewHeight(e, initialHeight, mouseStart, false);
				    if (selected !== null) {
				    	if(bottom < total){
				    		newHeight = newHeight < durationAPICtrl.sliderHeights.unit ? durationAPICtrl.sliderHeights.unit : newHeight;
				    	}else{
				    		newHeight = max > newHeight ? newHeight: max ;
				    	}
				    	durationAPICtrl.elements.duration.setHeight(newHeight);
				    }
				}
				
				
				function _destroy() {
				    selected = null;
				    mouseStart = 0;
				    durationAPICtrl.destroy();
				}
				
				element[0].onmousedown = _drag_init;
			}
		}
	});
	
})();
	