class FhirException {
    constructor(msg) {
        this.name = "FhirException";
        this.msg = msg;
    }
}

class Fhir {

    constructor(parsedJson) {
        this.obj = parsedJson;
    }

}

class Bundle extends Fhir {

    constructor(parsedJson) {
        super(parsedJson);
        this.bundle = this.obj;
    }

    validateProfile(profile) {
        try {
           var profiles = this.obj.meta.profile;
           if (profiles[0] == undefined || profiles[0] !== profile) {
               throw new FhirException("unsupported bundle profile " + profiles[0] );
           }
        } catch (error) {
            if (error.name == 'FhirException') {
                throw error;
            } else {
                throw new FhirException("unsupported bundle profile. " + error);
            }
        }
    }

    getUnwrappedResource(resource) {
        if (resource === undefined) {
            return undefined;
        } else if (resource.hasOwnProperty('resource') == true) {
            return resource.resource;
        } else {
            return resource;
        }
    }

    getEntryResourceByTypeFirstRep(type) {
        let array = this.obj.entry;
        for (let index = 0; index < array.length; index++) {
            let resource = this.getUnwrappedResource(array[index]);
            if (resource !== undefined) {
                if (resource.resourceType == type) {
                    return resource;
                }
            }
        }
        return undefined;
    }

    getEntryResourceByTypeFirstRep(type, profile) {
        let array = this.obj.entry;
        for (let index = 0; index < array.length; index++) {
            let resource = this.getUnwrappedResource(array[index]);
            if (resource !== undefined) {
                if (resource.resourceType == type) {
                    if (resource.meta !== undefined && resource.meta.profile == profile) {
                        return resource;
                    }
                }
            }
        }
        return undefined;
    }

    getEntryResourceByType(type, profile) {
        let resources = [];
        let array = this.obj.entry;
        for (let index = 0; index < array.length; index++) {
            let resource = this.getUnwrappedResource(array[index]);
            if (resource !== undefined) {
                if (resource.resourceType == type) {
                    if (profile !== undefined) {
                        if (resource.meta !== undefined && resource.meta.profile == profile) {
                            resources.push(resource);
                        }
                    } else {
                        resources.push(resource);
                    }
                }
            }
        }
        return resources;
    }

    get entry() {
        return this.obj.entry;
    }

}

class Quantity {

    constructor(r) {
        if (r === undefined) {
            this.value = undefined;
            this.unit = undefined;
        } else {
            this.value = r.value;
            this.unit = new Code();
            this.unit.id = r.id;
            this.unit.code = r.code;
            this.unit.display = r.unit;
        }
        if (this.unit == undefined || (this.unit.code == undefined && this.unit.display === undefined)) {
            this.display = this.value;
        } else if (this.unit !== undefined && this.unit.display !== undefined) {
            this.display = this.value + " " + this.unit.display; 
        } else if (this.unit !== undefined && this.unit.code !== undefined) {
            this.display = this.value + " " + this.unit.code;
        }
    }

}

class DataType {

    constructor() {
    }

    getReference(res, field, resourceType, resourceId) {
        if (field === undefined || resourceId === undefined) {
            return undefined;
        } else {
            let containeds = res.contained;
            if (containeds === undefined) {
                throw "failed to find contained resource for reference " + field;
            } else {
                for (let index = 0; index < containeds.length; index++) {
                    const contained = containeds[index];
                    if (contained.resourceType == resourceType && contained.id == resourceId) {
                        return contained;
                    }                    
                }
                throw "failed to find contained resource for reference " + field;
            }
        }
    }

    getExtension(res, url) {
        let extensions = res.extension;
        if (extensions == undefined) {
            return undefined;
        } else {
            let found = undefined;
            for (let index = 0; index < extensions.length; index++) {
                const ext = extensions[index];
                if (ext.url == url) {
                    if (found == undefined) {
                        found = ext;
                    } else {
                        throw "more than 1 extention found with url " + url;
                    }
                }
            }
            return found;
        }
    }

    getExtensionValue(res, url) {
        if (res === undefined) {
            return undefined;
        }
        let ext = this.getExtension(res, url);
        if (ext === undefined) {
            return undefined;
        } 
        else if (ext.hasOwnProperty('valueBoolean')) {
            return ext.valueBoolean;
        }
        else if (ext.hasOwnProperty('valueInteger')) {
            return ext.valueInteger;
        }
        else if (ext.hasOwnProperty('valueString')) {
            return ext.valueString;
        }
        else if (ext.hasOwnProperty('valueCodeableConcept')) {
             let value = ext.valueCodeableConcept;
             return new Code(value.coding[0])
        }
        return undefined;        
    }

    getExtensionValueCodeableConcept(ext) {
        if (ext === undefined) {
            return undefined;
        } else {
            return new Code(ext.coding[0]);
        }
    }

    getExtensionValueInteger(ext) {
        if (ext === undefined) {
            return undefined;
        } else {
            return ext.valueInteger;
        }
    }

    getExtensionValueString(ext) {
        if (ext === undefined) {
            return undefined;
        } else {
            return ext.valueString;
        }
    }
    
    getQuantity(value) {
        return new Quantity(value);
    }

    getCode(field) {
        if (field === undefined) {
            return undefined;
        }
        else if (field.hasOwnProperty('code')) {
            return this.getCode(field.code);
        }
        else if (field.hasOwnProperty('coding')) {
            return new Code(field.coding[0]);
        } 
        else {
            return new Code(field);
        }
    }

    getCodeOrText(field) {
        if (field === undefined) {
            return undefined;
        } 
        else if (field.hasOwnProperty('code')) {
            return this.getCodeOrText(field.code);
        }
        else if (field.hasOwnProperty('coding')) {
            return new CodeOrText(field);
        } 
        else {
            return new CodeOrText(field.coding[0]);
        }
    }

}

class Editable {

    constructor() {

    }

    getAsCode(obj) {
        if (obj === undefined) {
            return undefined;
        }
        else if (Code.prototype.isPrototypeOf(obj)) {
            return Object.assign({}, obj);
        }
        else if (CodeOrText.prototype.isPrototypeOf(obj)) {
            return Object.assign({}, obj.code);
        }
        else if (obj.coding) {
            return Object.assign({}, obj.coding);
        }
        else {
            return undefined;
        }
    }

    getReason(obj) {
        if (obj != undefined && obj.coding) {
            return { code: Object.assign({}, obj.coding[0]), display: obj.coding[0].display };
        }
    }

    getAsCodeOrText(obj) {
        if (obj === undefined) {
            return undefined;
        }
        else if (CodeOrText.prototype.isPrototypeOf(obj)) {
            return obj;
        }
        else if (Code.prototype.isPrototypeOf(obj)) {
            return { 
                code: Object.assign({}, obj), 
                text : obj.display,
                display: obj.display 
            };
        }
        else if (obj.coding) {
            return { 
                code: Object.assign({}, obj.coding), 
                text : obj.coding.display,
                display: obj.coding.display 
            };
        }
    }

    getAsCodeArray(obj) {
        if (obj === undefined) {
            return undefined;
        }
        if (Array.isArray(obj)) {
            return obj.map(function (e) { return getAsCode(e); });
        }
        return [this.getAsCode(obj)];
    }

    getAsInspqRefrence(obj) {
        if (obj === undefined) {
            return undefined;
        }
        else if (typeof obj == "string") {
            return { value: obj };
        }
        else if (obj.reference && obj.display) {
            return {value: obj.reference, display: obj.display};
        }
    }

}

class Resource extends DataType {

    constructor(res) {
        super()
        if (res === undefined) {
            this.res = {};
        }
        else if (res.hasOwnProperty('res')) {
            this.res = res.res;
        } 
        else {
            this.res = res;
        }
    }

    get id() {
        return this.res.id;
    }

    get versionId() {
        let meta = this.res.meta;
        if (meta == undefined) {
            return undefined;
        } else {
            return meta.versionId;
        }
    }
    
    get lastUpdated() {
        let meta = this.res.meta;
        if (meta == undefined) {
            return undefined;
        } else {
            return meta.lastUpdated;
        }
    }

    getPractitionerReference(field, resourceId) {
        if (resourceId === undefined) {
            return undefined;
        }
        if (resourceId.hasOwnProperty('reference')) {
            resourceId = resourceId.reference;
        }
        let r = this.getReference(this.res, field, 'Practitioner', resourceId);
        if (r === undefined) {
            return r;
        } else {
            return new Practitioner(r);
        }
    }

    getPatientReference(field, resourceId) {
        return this.getReference(this.res, field, 'Patient', resourceId);
    }

    getLocationReference(field, resourceId) {
        return this.getReference(this.res, field, 'Location', resourceId);
    }

    static setLocation(obj, newValue) {
        // $scope.condition.location = {};
        // $scope.condition.location.value = s.id;
        // $scope.condition.location.display = s.name;
        // $scope.condition.location.managingOrganization = s.managingOrganization;

        if (newValue !== undefined && newValue.hasOwnProperty('resource')) {
            newValue = newValue.resource;
        }
        if (newValue === undefined) {
            obj.locationValue = undefined;
            obj.location = undefined;
            obj.location.managingOrganization = undefined;
        } else if(newValue.value != undefined){
            obj.location = Object.assign(obj.location?obj.location:{}, newValue);
        } else {
            obj.locationValue = newValue.id;
            obj.location = Object.assign({}, newValue);
            obj.location.value = newValue.id;
            obj.location.display = newValue.name;
            obj.location.managingOrganization = newValue.managingOrganization;
        }
    }

}

class ValueSet extends Resource {
 
    constructor(res) {
        super()
        this.res = res;
    }

    getValues() {
         //e structure is {'code' : text, 'display' : text}
         var val = this;
         this.res.codeSystem.concept=  this.res.codeSystem.concept.map(function(e){
            e.version = val.res.codeSystem.version;
            e.system = val.res.codeSystem.system
            return e;
         })
        return this.res.codeSystem.concept;
    }

}

class Location2 extends DataType {

    constructor(id, display, contained) {
        super();
        this.id = id;
        this.name = contained.name;
        if (display === undefined) {
            this.display = this.name;
        } else {
            this.display = display;
        }
        this.status = contained.status;
        this.managingOrganization = contained.managingOrganization;
    }

}

class PatientContact extends DataType {
    constructor(r) {
        super();
        this.id = r.id;
        this.relationship = this.computeRelationship(r);
        this.firstName = this.computeFirstName(r);
        this.lastName = this.computeLastName(r);
    }

    computeRelationship(r) {
        if (r.relationship !== undefined) {
            if (Array.isArray(r.relationship)) {
                if (r.relationship.length > 0) {
                    return this.getCode(r.relationship[0]).display;
                }
            } else {
                return this.getCode(r.relationship).display;
            }
        }
        return undefined;
    }

    computeFirstName(r) {
        let array = r.name.given;
        if (array === undefined) {
            return '';
        } else {
            return array.join(' ');
        }
    }

    computeLastName(r) {
        let array = r.name.family;
        if (array === undefined) {
            return '';
        } else {
            return array.join(' ');
        }
    }
}

class Patient extends Resource {

    constructor(res) {
        super(res);

        this.matchRamq = this.getExtensionValue(this.res, 'http://www.santepublique.rtss.qc.ca/sipmi/fa/1.0.0/extensions/#patient/matchramq');

        this.firstName = this.computeFirstName();
        this.lastName = this.computeLastName();
        this.fullName = this.computeFullName(this.firstName, this.lastName);
        this.birthDate = this.res.birthDate;
        this.gender = this.computeGender();
        this.namQc = this.computeNamQc();

        this.deceased = this.res.deceasedBoolean === undefined ? false : this.res.deceasedBoolean;
        this.active = this.res.active;
        // this.active = true;

        this.address = this.res.address;
        this.telecoms = this.res.telecom;
        this.contacts = this.computeContacts();

    }

    computeFullName(firstName, lastName) {
        if (firstName === undefined && lastName === undefined) {
            return undefined;
        }
        else if (firstName !== undefined && lastName !== undefined) {
            return lastName + ", " + firstName;
        }
        else if (firstName !== undefined) {
            return firstName;
        }
        else {
            return lastName;
        }
    }

    computeContacts() {
        let array = [];
        if (this.res.contact !== undefined) {
            for (let index = 0; index < this.res.contact.length; index++) {
                const c = this.res.contact[index];
                array.push(new PatientContact(c));
            }
        }
        return array;
    }

    computeNamQc() {
        let identifiers = this.res.identifier;
        if (identifiers === undefined) {
            return undefined;
        } else if (Array.isArray(identifiers)) {
            for (let index = 0; index < identifiers.length; index++) {
                let identifier = identifiers[index];
                if (identifier.system == 'http://www.santepublique.rtss.qc.ca/sipmi/fa/1.0.0/vocabulary/identifierType?code=NAM') {
                    let healthCardOrigin = this.getExtensionValue(identifier, 'http://www.santepublique.rtss.qc.ca/sipmi/fa/1.0.0/extensions/#patient/healthcardorigin');
                    if (healthCardOrigin !== undefined && healthCardOrigin.code == 'QC') {
                        let type = this.getCode(identifier.type);
                        if (type !== undefined && type.code == 'NAM') {
                            return identifier.value;
                        }
                    }
                }
            }
            return undefined;
        }
    }

    computeGender() {
        if (this.res.gender == "male") {
            return "H";
        } else if (this.res.gender == "female") {
            return "F"
        } else {
            return undefined;
        }
    }

    computeFirstName() {
        let array = this.res.name[0].given;
        if (array === undefined) {
            return '';
        } else {
            return array.join(' ');
        }
    }

    computeLastName() {
        let array = this.res.name[0].family;
        if (array === undefined) {
            return '';
        } else {
            return array.join(' ');
        }
    }

}

class Practitioner extends Resource {

    constructor(res) {
        super(res);
        if (this.res.identifier !== undefined) {
            this.identifier = this.res.identifier[0].value;
        } else{
            this.identifier = undefined;
        }
        this.active = this.res.active;
        if (this.res.practitionerRole !== undefined) {
            this.practitionerRole = this.res.practitionerRole[0].role.coding[0];
        } else {
            this.practitionerRole = undefined;
        }
        this.value = this.id;

        if (this.res.name == undefined) {
            this.display = "Inconnu";
        }
        else if (this.res.name.given == undefined && this.res.name.family == undefined) {
            this.display = "Inconnu";
        }
        else if (this.res.name.given != undefined && this.res.name.family !== undefined) {
            this.display = this.res.name.given.concat(this.res.name.family).join(" ");
        }
        else if (this.res.name.given !== undefined) {
            this.display = this.res.name.given;
        }
        else {
            this.display = this.res.name.family;
        }

        
    }

}

class CodeOrText {

    constructor(r) {
        if (r.hasOwnProperty('code')) {
            this.text = r.display;
            this.display = r.display;
            this.code = new Code(r);
        } else {
            this.text = r.text;
            this.display = r.text;
            if (r.coding !== undefined) {
                this.code = new Code(r.coding[0])
                this.display = this.code.display;
            } else {
                this.code = undefined;
            }
        }
    }

}

class Code {

    constructor(coding) {
        if (coding !== undefined && coding.hasOwnProperty('coding')) {
            coding = coding['coding'];
        }
        if (coding !== undefined && coding.hasOwnProperty('code')) {
            if (typeof coding.code === 'object') {
                coding = coding.code;
            }
        }
        if (Array.isArray(coding)) {
            coding = coding[0];
        }
        if (coding === undefined) {
            this.id = undefined;
            this.code = undefined;
            this.display = undefined;
            this.system = undefined;
            this.versionId = undefined;
        }
        else {
            this.id = coding.id;
            this.code = coding.code;
            this.display = coding.display;
            this.system = coding.system;
            this.version = coding.version;
            this.extension = coding.extension;
        }
    }
}

class Annotation {

}

class Condition extends Resource {
    constructor(res) {
        super(res);
        this.locationValue = this.getExtensionValue(res, "http://www.santepublique.rtss.qc.ca/sipmi/fa/1.0.0/extensions/#condition/location");
        this.antigen = this.getExtensionValue(res, "http://www.santepublique.rtss.qc.ca/sipmi/fa/1.0.0/extensions/#condition/antigen");
        this.agent = this.getExtensionValue(res, "http://www.santepublique.rtss.qc.ca/sipmi/fa/1.0.0/extensions/#condition/agent");
        this.reason = this.getCode(res.code);
        this.otherReason = this.getExtensionValue(this.reason, "http://www.santepublique.rtss.qc.ca/sipmi/fa/1.0.0/extensions/#condition/otherreason");
        
        //category is kept as a plain string and not modified afterward
        this.category = this.getCode(res.category);
        if (this.category !== undefined) {
            this.category = this.category.code;
        }

        this.onsetPeriodStart = res.onsetPeriod == undefined ? undefined : res.onsetPeriod.start;
        this.onsetPeriodEnd = res.onsetPeriod == undefined ? undefined : res.onsetPeriod.end;
        
        if (res.evidence === undefined) {
            this.evidence = undefined;
        } else{
            this.evidence = this.getCode(res.evidence[0]);
        }

        this.note = {
            text : res.notes
        };

        //is lazy set by controller
        this.managingOrganization = undefined;
        this._updateIsImmunizationProof();
    }
    
    setLocation(res) {
        Resource.setLocation(this, res);        
    }

    _updateIsImmunizationProof() {
        this.isImmunizationProof = this.category === 'immunizationProof';
    }

    setAsContraindicationPrecaution() {
        this.category = 'contraindicationPrecaution';
        this._updateIsImmunizationProof();
    }
    
    setAsImmunizationProof() {
        this.category = 'immunizationProof';
        this._updateIsImmunizationProof();
    }

    compare(other) {
        if (this.onsetPeriodStart == other.onsetPeriodStart) {
            return 0;
        } else if (this.onsetPeriodStart > other.onsetPeriodStart) {
            return -1;
        } else {
            return 1;
        }
    }

    editable() {
        return new EditableCondition(this);
    }
}

class EditableCondition extends Editable {

    // faTransactionId;
    // id;
    // version;
    // identifier;
    // agent;
    // antigen;
    // reason;
    // otherReason;
    // category;
    // location;
    // evidence;
    // note;
    // onsetPeriodStart;
    // onsetPeriodEnd;

    constructor(condition) {
        super();

        this.faTransactionId = undefined;

        const id = condition.id;
        if (id != undefined) {
            this.id = parseInt(id)
        } else {
            this.id = undefined;
        }

        const versionId = condition.versionId;
        if (versionId != undefined) {
            this.versionId = parseInt(versionId)
        } else {
            this.versionId = undefined;
        }

        this.otherReason = condition.otherReason;
        this.category = condition.category;
        this.isImmunizationProof = condition.isImmunizationProof;
        this.agent = this.getAsCode(condition.agent);
        this.antigen = this.getAsCode(condition.antigen);
        this.reason = this.getReason(condition.res.code);
        this.evidence = this.getAsCodeOrText(condition.evidence);
        
        this.locationValue = condition.locationValue;
        this.location = Object.assign({}, condition.location);
        // res.location = getAsInspqRefrence(this.location);

        if (condition.onsetPeriodStart != undefined) {
            this.onsetPeriodStart = moment(condition.onsetPeriodStart).format("YYYY-MM-DD");
        } else {
            this.onsetPeriodStart = undefined;
        }
        if (condition.onsetPeriodEnd != undefined) {
            this.onsetPeriodEnd = moment(condition.onsetPeriodEnd).format("YYYY-MM-DD");
        } else {
            this.onsetPeriodEnd = undefined;
        }

        //always start with a blank note
        this.note = condition.note;
    }

    setAgent(newValue) {
        this.agent = newValue;
        this.antigen = undefined;
    }

    setAntigen(newValue) {
        this.antigen = newValue;
        this.agent = undefined;
    }

    setLocation(value) {
        Resource.setLocation(this, value);
    }
    
}

class Flag extends Resource {
    constructor(res)  {
        super(res);
        this.code = this.getCodeOrText(res.code);
        this.category = this.getCodeOrText(res.category);

        this.antigen = this.code;
        this.reason = this.category;

        if (res.period === undefined) {
            this.periodStart = undefined;
            this.periodEnd = undefined;
        } else {
            this.periodStart = res.period.start;
            this.periodEnd = res.period.end;
        }
    }

    compare(other) {
        if (this.periodEnd !== other.periodEnd) {
            if (this.periodEnd === undefined) {
                return -1;
            }
            else if (other.periodEnd === undefined) {
                return 1;
            }
        }
        if (this.periodStart == other.periodStart) {
            return 0;
        } else if (this.periodStart > other.periodStart) {
            return -1;
        } else {
            return 1;
        }
    }

    editable() {
        return new EditableRefusal(this);
    }

}

class EditableRefusal extends Editable {

    constructor(refusal) {
        super();
        // faTransactionId;
        // id;
        // version;
        // identifier;
        // agent;
        // antigens;
        // reason;
        // periodStart;
        // periodEnd;

        this.faTransactionId = undefined;
        
        if (refusal.id !== undefined) {
            this.id = parseInt(refusal.id)
        } else {
            this.id = undefined;
        }
        
        this.versionId = undefined;
        if (refusal.versionId !== undefined) {
            this.versionId = parseInt(refusal.versionId)
        }
        this.antigens = this.getAsCodeArray(refusal.antigen);
        this.reason = this.getAsCode(refusal.category.code);
        
        if (refusal.periodStart != undefined) {
            this.periodStart = moment(refusal.periodStart).format("YYYY-MM-DD");
        } else {
            this.periodStart = undefined;
        }
        
        if (refusal.periodEnd != undefined) {
            this.periodEnd = moment(refusal.periodEnd).format("YYYY-MM-DD");
        } else {
            this.periodEnd = undefined;
        }

        this.note = refusal.note;
    }

}

class NewRefusal {

    constructor() {
        this.periodStart = moment().format("YYYY-MM-DD");
        this.allAntigen = 'true';
        this.antigens = [];
        this.reason = undefined;
    }

}

class RecommandationDate extends DataType {

    constructor(dateCriterion) {
        super();
        this.date = dateCriterion.value;
        this.dateQualifier = this.getCodeOrText(dateCriterion.code);
    }
    
    //qualifier code can be : due, recommended, earliest, overdue, latest
    getQualifierCode() {
        if (this.dateQualifier.code === undefined) {
            return '';
        } else if (this.dateQualifier.code.code === undefined){
            return '';
        } else {
            return this.dateQualifier.code.code;
        }
    }

}

class RecommandationVaccineCode extends DataType {
    constructor(vaccineCodeNode) {
        super();
        let coding = vaccineCodeNode.coding[0];
        this.id = coding.id;
        this.code = coding.code;
        this.display = coding.display;
        this.isVaccine = this.getExtensionValue(coding, 'http://www.santepublique.rtss.qc.ca/sipmi/fa/1.0.0/extensions/#immunizationrecommandation/vaccinetype') == true;
    }
}

class Recommendation extends DataType {
    
    constructor(res) {
        super();
        this.res = res.recommendation[0];

        //The date the immunization recommendation was created.
        this.createdDate = this.res.date;

        //Vaccine that pertains to the recommendation.
        this.vaccineCode = new RecommandationVaccineCode(this.res.vaccineCode);

        //This indicates the next recommended dose number (e.g. dose 2 is the next recommended dose).
        this.doseNumber = this.res.doseNumber;

        //Vaccine administration status
        this.forecastStatus = this.getCodeOrText(this.res.forecastStatus);

        //Vaccine date recommendations.  For example, earliest date to administer, latest date to administer, etc.
        this.dates = [];
        for (let index = 0; index < this.res.dateCriterion.length; index++) {
            const c = this.res.dateCriterion[index];
            this.dates.push(new RecommandationDate(c));
        }

        //compute eligible date
        //qualifier code can be : due, recommended, earliest, overdue, latest
        this.eligibleDate = this.findEarliestDate();
        this.plannedDate = this.findDateByQualifier('due');
    }

    compare(other) {
        if (this.eligibleDate == other.eligibleDate) {
            return 0;
        } else if (this.eligibleDate > other.eligibleDate) {
            return -1;
        } else {
            return 1;
        }
    }

    findEarliestDate() {   
        return this.findDateByQualifier('earliest');
    }

    findDateByQualifiers(qualifiers) {
        for (let index = 0; index < qualifiers.length; index++) {
            const qualifier = qualifiers[index];
            let date = this.findDateByQualifier(qualifier);
            if (date !== undefined) {
                return date;
            }
        }
        return undefined;
    }

    findDateByQualifier(qualifier) {
        for (let index = 0; index < this.dates.length; index++) {
            const date = this.dates[index];
            if (date.dateQualifier.code.code == qualifier) {
                return date.date;
            }
        }
        return undefined;
    }

}

class AntigenStatus extends Resource {

    constructor(res) {
        super(res);
    }

    get antigen() {
        return this.getExtensionValueCodeableConcept("antigen");
    }

    get doseNumber() {
        return this.getExtensionValueInteger("doseNumber");
    }

    get status() {
        return this.getExtensionValueCodeableConcept("status");
    }

}

class Immunization extends Resource {

    constructor(res) {
        super(res);

        if (this.res.date === undefined) {
            this.date = undefined;
        } else {
            this.date = moment(this.res.date).format("YYYY-MM-DD");
        }
        
        this.tradeName = this.computeTradeName();
        this.tradeNameIsUnknown = this.tradeName === undefined;
        this.vaccineCode = this.getCode(this.res.vaccineCode);
        this.status = this.res.status;
        this.patient = this.getPatientReference(this.res.patient);
        this.wasNotGiven = this.res.wasNotGiven;
        this.reported = this.res.reported;
        this.lotNumber = this.res.lotNumber;
        this.lotId = this.computeLotId();
        this.expirationDate = this.res.expirationDate;
        this.notes = this.computeNotes();
        this.performer = this.getPractitionerReference('performer', this.res.performer);
        this.site = this.getCode(this.res.site);
        this.route = this.getCode(this.res.route);
        this.reason = this.computeReason();
        this.location = this.computeLocation();
        this.doseQuantity = this.getQuantity(this.res.doseQuantity);
    }
    
    get doseQuantityValue() {
        if (this.doseQuantity === undefined) {
            return undefined;
        }
        return this.doseQuantity.value;
    }

    set doseQuantityValue(value) {
        if (this.doseQuantity === undefined) {
            this.doseQuantity = new Quantity();
        }
        this.doseQuantity.value = value;
    }

    get doseQuantityUnit() {
        if (this.doseQuantity === undefined) {
            return undefined;
        }
        return this.doseQuantity.unit;
    }

    set doseQuantityUnit(value) {
        if (this.doseQuantity === undefined) {
            this.doseQuantity = new Quantity();
        }
        this.doseQuantity.unit = value;
    }

    computeLocation() {
        let location = this.res.location;
        if (location === undefined) {
            return undefined;
        } else {
            let resourceId = undefined;
            if (location.id !== undefined) {
                resourceId = location.id;
            } else if (location.reference !== undefined) {
                resourceId = location.reference;
            }
            let field = location;
            let contained = this.getLocationReference(field, resourceId);
            return new Location2(resourceId, location.display, contained);
        }
    }

    computeNotes() {
        let r = this.res.note;
        let res = []
        if (Array.isArray(r)) {
            res = r;
        } else if(r != undefined){
            res = [r];
        }
        return res.map(function(e){
            if(e != undefined){
                let elem = angular.copy(e);
                if(elem.time != undefined){
                    elem.time = moment(elem.time).format("YYYY-MM-DD HH:mm:ss");
                }
            }
            return e;
        });
    }

    explanation() {
        let r = this.res.explanation;
        if (r === undefined) {
            return undefined;
        } else {
           return r;
        }
    }

    computeReason() {
        let r = this.explanation();
        if (r === undefined) {
            return undefined;
        } else {
           return new Code(r.reason[0]);
        }
    }

    get explanationReasonNotGiven() {
        let r = this.explanation();
        if (r === undefined) {
            return undefined;
        } else {
           return new CodeOrText(r.reasonNotGiven);
        }
    }

    get reactions() {
        let r = this.res.reaction;
        if (r === undefined) {
            return [];
        } else if (Array.isArray(r)) {
            return r;
        } else {
            return [r];
        }
    }

    get vaccinationProtocol() {
        let r = this.res.vaccinationProtocol;
        if (r === undefined) {
            return [];
        } else if (Array.isArray(r)) {
            return r;
        } else {
            return [r];
        } 
    }

    computeLotId() {
        let ext = this.getExtension(this.res, 'http://www.santepublique.rtss.qc.ca/sipmi/fa/1.0.0/extensions/#immunization/lotid');
        if (ext == undefined) {
            return undefined;
        } else {
            return ext.valueString;
        }
    }

    get reasonForDeletion() {
        let ext = this.getExtension(this.res, 'http://www.santepublique.rtss.qc.ca/sipmi/fa/1.0.0/extensions/#immunization/reasonfordeletion');
        if (ext == undefined) {
            return undefined;
        } else {
            return ext;
        }
    }

    get otherReasonForDeletion() {
        let ext = this.getExtension(this.res, 'http://www.santepublique.rtss.qc.ca/sipmi/fa/1.0.0/extensions/#immunization/otherreasonfordeletion');
        if (ext == undefined) {
            return undefined;
        } else {
            return ext;
        }
    }

    get overrideStatus() {
        let ext = this.getExtension(this.res, 'http://www.santepublique.rtss.qc.ca/sipmi/fa/1.0.0/extensions/#immunization/overridestatus');
        if (ext == undefined) {
            return undefined;
        } else {
            return new Code(ext.valueCodeableConcept.coding[0]);
        }
    }

    computeTradeName() {
        let ext = this.getExtension(this.res, 'http://www.santepublique.rtss.qc.ca/sipmi/fa/1.0.0/extensions/#immunization/tradename');
        if (ext == undefined) {
            return undefined;
        } else {
            return ext.valueString;
        }
    }    

    get overrideReason() {
        let ext = this.getExtension(this.res, 'http://www.santepublique.rtss.qc.ca/sipmi/fa/1.0.0/extensions/#immunization/overridereason');
        if (ext == undefined) {
            return undefined;
        } else {
            return ext;
        }
    }

    get antigenStatus() {
        let ext = this.getExtension(this.res, 'http://www.santepublique.rtss.qc.ca/sipmi/fa/1.0.0/extensions/#immunization/antigenstatus');
        if (ext == undefined) {
            return undefined;
        } else {
            return AntigenStatus(res);
        }
    }

    compare(other) {
        if (this.date == other.date) {
            return 0;
        } else if (this.date > other.date) {
            return -1;
        } else {
            return 1;
        }
    }

    editable() {
        return Object.assign(new Immunization(), this);
    }
}

class XInspqImmunization {

    constructor(imm) {
        this.updatePatientProfile = false;
        this.faTransactionId = imm.faTransactionId;

        this.id = imm.id;
        this.versionId = imm.versionId;
        this.agent = this.getAsXInspqCodeOrText(imm.vaccineCode);
        this.tradeName = imm.tradeName;
        this.tradeNameIsUnknown = imm.tradeNameIsUnknown;
        this.lotNumber = imm.lotNumber;
        this.lotId = imm.lotId;
        this.date = imm.date;
        this.doseQuantity = imm.doseQuantity;
        this.doseQuantityIsUnknown = imm.doseQuantityUnknown,
        this.site = this.getAsXInspqCode(imm.site);
        this.route = this.getAsXInspqCode(imm.route);
        this.reason = this.getAsXInspqCode(imm.reason);
        this.location = this.getAsXInspqReference(imm.location);
        this.performer = this.getAsXInspqReference(imm.performer);
       
        this.notes = [];
        if (imm.comment !== undefined) {
            this.notes.push({text : imm.comment});
        }
    }

    addNote(value) {
        this.notes.push({text : value});
    }

    getAsXInspqCode(value) {
        if (value === undefined) {
            return undefined;
        }
        else if (Code.prototype.isPrototypeOf(value)) {
            return {
                id : value.id,
                code : value.code,
                display : value.display,
                system : value.system,
                version : value.version,
            }
        }
        else if (CodeOrText.prototype.isPrototypeOf(value)) {
            return this.getAsXInspqCode(value.code);
        }
        else if (value.coding !== undefined) {
            return this.getAsXInspqCode(new Code(value.coding));
        }
        else {
            throw "unsupported value for conversion to XInspqCode"
        }
    }

    getAsXInspqCodeOrText(value) {
        if (value === undefined) {
            return undefined;
        }
        else if (CodeOrText.prototype.isPrototypeOf(value)) {
            return new CodeOrText(value);
        }
        else if (Code.prototype.isPrototypeOf(value)) {
            return new CodeOrText({code : value});
        }
        else if (value.coding) {
            return new CodeOrText({code : new Code(value.coding)});
        }
    }

    getAsXInspqReference(value) {
        if (value === undefined) {
            return undefined;
        }
        if (value.hasOwnProperty('resourceType')) {
            return {value : value.id, display : value.name};
        }
        if (Practitioner.prototype.isPrototypeOf(value)) {
            return {value : value.id, display : value.display};
        }
        if (Location2.prototype.isPrototypeOf(value)) {
            return {value : value.id, display : value.display};
        }
        throw "unsupported value for getAsXInspqReference";
    }
}

class ImmunizationProfile extends Bundle {

    constructor(bundle) {
        super(bundle);
        this.validateProfile("http://www.santepublique.rtss.qc.ca/sipmi/fa/1.0.0/profiles/InspqImmunizationProfileBundle.structuredefinition.xml");
        this.patient = this.computePatient();
        this.immunizations = this.computeImmunizations();
        this.conditions = this.computeConditions();
        this.flags = this.computeFlags();
        this.recommandations = this.computeRecommandations();

        this.canAddImmunization = this.isPatientFusionFlagOff();
        this.canAddCondition = this.isPatientFusionFlagOff();
        this.canAddFlag = this.isPatientFusionFlagOff();

        this.sortLists();

        this.needReload = false;
    }

    canPrint() {
        return this.patient !== undefined && this.needReload === undefined || this.needReload == false;
    }

    needReload() {
        return this.needReload;
    }

    sortLists() {
        this.sortImmunizations();
        this.sortConditions();
        this.sortFlags();
        this.sortRecommendations();
    }

    sortImmunizations() {
        if (this.immunizations !== undefined) {
            this.immunizations.sort((a, b) => a.compare(b));
        }
    }

    sortConditions() {
        if (this.conditions !== undefined) {
            this.conditions.sort((a, b) => a.compare(b));
        }
    }

    sortFlags() {
        if (this.flags !== undefined) {
            this.flags.sort((a, b) => a.compare(b));
        }
    }

    sortRecommendations() {
        if (this.recommandations !== undefined) {
            this.recommandations.sort((a, b) => a.compare(b));
        }
    }

    isPatientFusionFlagOn() {
        if (this.patient === undefined) {
            return false;
        }
        return this.patient.active == false;
    }

    isPatientFusionFlagOff() {
        if (this.patient === undefined) {
            return true;
        }
        return this.patient.active == true || this.patient.active === undefined;
    }

    remove(res) {
        this.needReload = true; //updating does not ensure profile is all up to date, only reloading does this.

        //if res is a batch response bundle
        if (res.resourceType == "Bundle") {
            let entryArray = res.entry;
            for (let index = 0; index < entryArray.length; index++) {
                const entry = entryArray[index];
                const resource = entry.resource;
                this.removeResource(resource);
            }
        }
        else {
            this.removeResource(res);
        }
        this.sortLists();
    }

    //return newly created or modified objects
    update(res) {
        this.needReload = true; //updating does not ensure profile is all up to date, only reloading does this.

        //if res is a batch response bundle
        let newObjects = [];
        if (res.resourceType == "Bundle") {
            let entryArray = res.entry;
            for (let index = 0; index < entryArray.length; index++) {
                const entry = entryArray[index];
                const resource = entry.resource;
                let r = this.updateResource(resource);
                if (r !== undefined) {
                    newObjects.push(r);
                }
            }
        }
        else {
            let r = this.updateResource(res);
            if (r !== undefined) {
                newObjects.push(r);    
            }
        }
        this.sortLists();
        return newObjects;
    }

    //return newly created or modified object
    updateResource(resource) {
        if (resource.resourceType == "Flag") {
            let flag = new Flag(resource);
            this.updateArrayById(this.flags, flag);
            return flag;
        } 
        else if (resource.resourceType == "Immunization") {
            let imm = new Immunization(resource);
            this.updateArrayById(this.immunizations, imm);
            return imm;
        }
        else if (resource.resourceType == "Condition") {
            let cond = new Condition(resource);
            this.updateArrayById(this.conditions, cond);
            return cond;
        } 
        else if (resource.resourceType == "Patient") {
            //ignore
            return undefined;
        } 
        else {
            throw "ImmunizationProfile.update - unsupported resource type" + resource.resourceType;
        }
    }

    removeResource(resource) {
        if (resource.resourceType == "Flag") {
            let flag = new Flag(resource);
            this.removeFromArrayById(this.flags, flag);
        } 
        else if (resource.resourceType == "Immunization") {
            let imm = new Immunization(resource);
            this.removeFromArrayById(this.immunizations, imm);
        }
        else if (resource.resourceType == "Condition") {
            let cond = new Condition(resource);
            this.removeFromArrayById(this.conditions, cond);
        } 
        else if (resource.resourceType == "Patient") {
            //ignore
            return undefined;
        } 
        else {
            throw "ImmunizationProfile.update - unsupported resource type" + resource.resourceType;
        }
    }

    removeFromArrayById(array, newElement) {
        let found = false;
        const newElementId = newElement.id;
        for (let index = 0; index < array.length; index++) {
            const element = array[index];
            if (element.id == newElementId) {
                array.splice(index, 1);
                found = true;
                break;
            }
        }
        // if (found == false) {
            // array.push(newElement);
        // }
    }

    updateArrayById(array, newElement) {
        let found = false;
        const newElementId = newElement.id;
        for (let index = 0; index < array.length; index++) {
            const element = array[index];
            if (element.id == newElementId) {
                array[index] = newElement;
                found = true;
            }
        }
        if (found == false) {
            array.push(newElement);
        }
    }

    computePatient() {
        let bundleRes = this.getEntryResourceByTypeFirstRep('Patient', 'http://www.santepublique.rtss.qc.ca/sipmi/fa/1.0.0/profiles/InspqPatient.structuredefinition.xml');
        if (bundleRes == undefined) {
            return undefined;
        } else {
            return new Patient(bundleRes);
        }
    }

    computeImmunizations() {
        let bundleRes = this.getEntryResourceByTypeFirstRep('Bundle', 'http://www.santepublique.rtss.qc.ca/sipmi/fa/1.0.0/profiles/ImmunizationBundle');
        if (bundleRes == undefined) {
            return [];
        } else {
            let result = [];
            let resources = new Bundle(bundleRes).getEntryResourceByType('Immunization', 'http://www.santepublique.rtss.qc.ca/sipmi/fa/1.0.0/profiles/InspqImmunization.structuredefinition.xml');
            for (let index = 0; index < resources.length; index++) {
                const res = resources[index];
                result.push(new Immunization(res));
            }
            return result;
        }
    }

    computeConditions() {
        let bundleRes = this.getEntryResourceByTypeFirstRep('Bundle', 'http://www.santepublique.rtss.qc.ca/sipmi/fa/1.0.0/profiles/ConditionBundle');
        if (bundleRes == undefined) {
            return [];
        } else {
            let result = [];
            let resources = new Bundle(bundleRes).getEntryResourceByType('Condition', 'http://www.santepublique.rtss.qc.ca/sipmi/fa/1.0.0/profiles/InspqCondition.structuredefinition.xml');
            for (let index = 0; index < resources.length; index++) {
                const res = resources[index];
                result.push(new Condition(res));
            }
            return result;
        }
    }

    computeFlags() {
        let bundleRes = this.getEntryResourceByTypeFirstRep('Bundle', 'http://www.santepublique.rtss.qc.ca/sipmi/fa/1.0.0/profiles/FlagBundle');
        if (bundleRes == undefined) {
            return [];
        } else {
            let result = [];
            let resources = new Bundle(bundleRes).getEntryResourceByType('Flag', 'http://www.santepublique.rtss.qc.ca/sipmi/fa/1.0.0/profiles/InspqFlag.structuredefinition.xml');
            for (let index = 0; index < resources.length; index++) {
                const res = resources[index];
                result.push(new Flag(res));
            }
            return result;
        }
    }

    computeRecommandations() {
        let bundleRes = this.getEntryResourceByTypeFirstRep('Bundle', 'http://www.santepublique.rtss.qc.ca/sipmi/fa/1.0.0/profiles/ImmunizationRecommandationBundle');
        if (bundleRes == undefined) {
            return [];
        } else {
            let result = [];
            let resources = new Bundle(bundleRes).getEntryResourceByType('ImmunizationRecommendation');
            for (let index = 0; index < resources.length; index++) {
                const res = resources[index];
                result.push(new Recommendation(res));
            }
            return result;
        }
    }

    findImmunization(id) {
        let array = this.immunizations;
        for (let index = 0; index < array.length; index++) {
            const imm = array[index];
            if (imm.id === id) {
                return imm;
            }
        }
        return undefined;
    }

}

class EmptyImmunizationProfile extends ImmunizationProfile {
    constructor() {
        super({ 
            "resourceType": "Bundle",
            "meta": {
                "profile": ["http://www.santepublique.rtss.qc.ca/sipmi/fa/1.0.0/profiles/InspqImmunizationProfileBundle.structuredefinition.xml"]
            },
            "type": "collection",
            "entry":[]
        });
        this.canPrint = function(){ return false};
        this.canAddImmunization = true;
        this.canAddCondition = false;
        this.canAddFlag = false;
    }
}