
// Design docs:
// - The saved file should have the following structure
// ====================================================
// **NOTE: There is a few diffrences from the saved file and the working strokeHistory object.
// ** the paths object is structured diffrent to minimise database saved object size by redusing redundacies.
// {//File
//    drawings: [
//        {//
//            strokes:{ // stroke could have multiple paths
//                lineWidth: 1.5,
//                color: "#000",
//                brush: 1, //image to be used as brush (for highlighter)
//                paths:[ // Each path corresponds to a stroke
//                    {
//                        x:[],// for each x that is added y must be added also. length must always be the same.
//                        y:[],
//                    }
//                ]
//            }
//        },
//        { cmd: "clearAll"},
//    ]
// }
// 
//************* */

function Stylet(canvasId, persistenceInputId){
  var ST = this;
  ST.generateUIDNotMoreThan1million = function() {
		return ("00000" + (Math.random()*Math.pow(36,5) << 0).toString(36)).slice(-5)
	}
  var id = "st"+ST.generateUIDNotMoreThan1million();
  Stylet.all[id] = ST;
  var SELECTIONCLASS = "st-selected"
  var styletDOM = '<canvas id="${id}-stylet-canvasl0" class="stylet-canvas layer0"></canvas>'+
  '<canvas id="${id}-stylet-canvasl1" class="stylet-canvas layer1"></canvas>'+
  '<div class="${id} stylet-toolbar st-no-print" style="position: absolute; bottom: 0; "> '+
  '    <button type="button" class="stylet-btn extra" onclick="Stylet.commit(\'${id}\', \'undo\')">'+
  '        <img height="24" src="data:image/png;base64, iVBORw0KGgoAAAANSUhEUgAAAEAAAABACAYAAACqaXHeAAAEsklEQVR4Xu2baYgcRRTHf/tB8mFRQiCKoLCBBE0QRVRQ8yUaohtvQRGPRcVgRElA0UhCIAoe0RAQ1AVP8ELBCzzwQg3BhJweoKLGVSMkIqJB8EARlf9QvfS209Ovjp6Z9FgwzMC8qvfer7uqXlW9GgJuAdZQXm51Mh1E9t+/hoB/DOZLLqYsAA5zn8MLvw8EvgV2576z3xtilFrq1gXgAOAc4FzgTGCGxZg2MvuAF4DXAMH4MbCd0mqpAVwKLHafUKfLjP3TgXgVeCgViFQARoEbgYWpDKtoZzMwDjwVqy8WwGzn+NJYQwLrv+1AqJsElRgAeuI3AQcHaU5b6XngcuBX32ZDAAwDTwDn+yqrWX67G3S/89HjC0BP+2ngVB8lXZT9HtCU+5lVpw+AWcCbgPp9v5cTga0WI60ATgI2AprfQ8vnwA7gS2BX7vuvQoCkQOkM4IRQRa7eXMubYAUQaov6o+btx4D3PBvRm5bFFPoOKdOBnztVrAvAuy5YeRbQE44tFwDXuf7t09YWQG9vaUkN4HfgNuAOHys9ZK8EbgCO8qjzAHBNmXxKAC8757d5GBciOsd1qY5PttDw7cDqdspSAShVEOKhsc4bwGlGWYmd4hZUU6qkAHAfsMzDkJSiVXsZeV0vuUApKYBHgCUpPQpoSw9AA6SljAFP5gVj3gCtxC6zaO2CzDPARQY97wMnA39ksjEADPr+I1LX9tqhwOvA0QajVgJrewVAemO318p81BT5qAHAF8ARTQQgnxSAaTFUVbRx8072NCybolUN+vxf1xsgGxQxKvqsKncDNzcRgHzSBqq26DqVD4FjmwpgBXBX1SvglvUT3Z4F6hwEM5/nAZ8YACwH7m0iAPn+kWFKvAe4XgB8wkkD2I4idcUBRaUWnxQRjtU5IsfCiql/FfBwRQNaTI02FcDpLjLsxGAncHxTAVgGQh3AjjQVwEFVe4HAb8Dw/wBiRpo+rjvwXWDgB8GBnwYHPhDyCoX7eCwLMs0yAKphbaSON3EatC6HddT/QxMBWDZE3soOVZoGwLol1nr9u7E5EdSJIypZNkUVAo/o9W8aAOu2uA5RLs4gN6UL+ByMnOWSNloMmgLAejT2HHBhvos1AYDP4aiy2zROTJb9HYAl5M2cfdwlU04ZY2MBZAZ0a7Mzb7xvgkTb1LkYAEX6L7rU2YmIacxSNSRF5k5gVbvGQwHohokAFIucV/6wYNRRQpKkWtvfZcaEAChzPq9DCVMPAq8kohCaJrcHOBL4JRUAi/NFEIKxCfjUE0aKRMnKbFGfN0A3QGKurHzlskU/AL7Jff6uKVX2EpfY3ZG7DwClyX8NzPR8kr0Qn1zsVCn3AaC25gP3A8dUNdyj/3XJ6jyX2G0ywReAGp0GrOthbmCZY0p5UZj7k8lzJxQCIGv/CpeI0A9XZta7u0s+vrdkYwCovrrCtcDV3prTVFB4q40N0+WIdipjAWRtHucgdAuEVnVyfMrCJoRpKgBFEApcUl+c3AtoL09ZYLqEkaSkBpAZpdPZsxPcIlVSo8JqOa6nrZghaakLQNHIRS6BUQNm9jnE/daNEp3V5y9P6zL1x8Zkpygg3QIQZWSdlf8Fv53nQTFtdscAAAAASUVORK5CYII="/>'+
  '    </button> '+
  '    <button type="button" class="stylet-btn extra" onclick="Stylet.commit(\'${id}\', \'redo\')">'+
'         <img height="24" src="data:image/png;base64, iVBORw0KGgoAAAANSUhEUgAAAEAAAABACAYAAACqaXHeAAAFDklEQVR4Xt2bWcitUxjHfwd3ZEoIEUUyXsiRQmaZisw5SHI6HcONhAvDiWQqF6ZyIfOQzC7IXGZJZkrIkFnGGyn6fb2v3ra99xre9b57f+9Tq/197fVM/72GZ63nWUsYNl0MXDLFxVVLevJ/e2AHYHNgi8anf2vDl8BXjVb//wzwdwsb/wnxdgXAasA+wAHAkcA2IUOmfH8/8BDwNPBDopw/gDWn8ZQG4FDgmMrxTRKNjeleA3Ef8FMEw3fAhn0A4K+9Ejg6wqgSXT4CrgRuDQj7FNiySwB2qxw/uYRXGTIeqIB4YwLve4Drz0RqMwWuAc7JMLo0y18VCI6IP0eEvwYsLQ3A+oAL076lPWkp7y1gGfBBQ86z1WJcbATsBTwMrNfS2K7Y3weObYDwGHBYqRHgIndDV5YXlNsE4R7g+BIAnADc3dJIp83LwBeN9jOw9kjbHTgI2K+FvhqEs4AVbQHYFvgw0xj3baeMn78nyti4iieMKw5P5LW7IDgFzm8DwFqAe+6miQY8AlwNvJTIN6m783h5BhCC0GobvKNaWWP9+Bi4CrglliGxn0B4uNklkS9rF7gcuCBBkXPchfLHBJ6crsb2Lm450+J/+iYFQkZ4ryZYp/NuP32SI+3ctgonAXAbEBveGo+f2taQTP4DgSczeRfYxgHgwcYIKob8Fc6L6dhhH/VfkSt/HAAO55hTnSv9EbmKC/N5PM6agqMAeJ5/PMI4V/tdM/b2CNELq7xXWb9NaMYTv1b3AQZS3gvYbgS2jlHQ7DMKwO3ASRFC9ge8ruqCgtdYJZU2AVgd+BrYKKDgzkiQcu2cGQDuq49GWH0w8EREv9wuMwPgptDBAXiuh3uAmQEQvD8DTuswzK1HzEwA2BF4J2LMbgV8FtGvTZeZAHAccG/Aarc+j8Zd00wAMKY2qptGAuTFSNdUxwFd61H+f6mx64EzAhrPBq7rw6o+ddRxgNtf6Hi5J/Bin8b1oasG4G1gp4BCv3+3D6P61FED8AuwTkCxmVwvNAdFKQCsWx1CBglAzBRok0abW9BSFsHtWlyPzz0AMduglyRmYwdF9QiICYQuBC4blPeNO8GYUNjU2IlDBcDsicUE08gtcGfALXMw1FzZPeyEiplMNrpeDIaaAHi1HLriNlli9nYw1ATAdLSlaCE6Cngw1GmxfN8EwNo+CxRD5W1PAWZkBkGj0Z1prlMiPBvMWpCbGHFH2KMaMRF4zW+XNqkxdwNHwqKmtsnRVYFq7LkHp0R6fFGDUKpAYhYgmEDdG3ihzSicdsZPLYV1BxGIzzse94bjlu8c0tCT/QOELjnMAKeUxOq8xoSquHMxOr1yfoMxArJACAFgXfAnGaWxAmAWuVQKPbZMLhmEEAACbX2w8yyHXm8USlpvmEqxjjflJoEQA4DCS9QJvwnYXgGeB75plMh6I23JrDlKj+aeSq322CwVsaq/r0SintfEAqDcEvXCmf4ksem4r0RG3w6MFZICgAJMjnpiTC2dTfKgRWfXK0tqrWOIolQAFGr9sMUUPk6YJ/LC9kzg2xSjcgCo5aeW0qbYldr3UuCiVCb7twFA/lk/mnJRvRa4K8f5EgDUevt+NqfjN1ct1/cFvrYjYFS5hZZesXtjFCq3yzG8mOO18tIA1HKtOTRWt/n8ZerjxQASxgwGYn7ailJXAIwaaYBjbnHc4+k1gO8Bn7n6WTff/BlKR+3nuaj0BUCufZ3z/QveCOgnaovQ/QAAAABJRU5ErkJggg=="/>'+
  '    </button>'+
  '    <button type="button" class="stylet-btn extra" onclick="Stylet.commit(\'${id}\', \'toggleLines\')">'+
  '        <img height="24" src="data:image/png;base64, iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAYAAADgdz34AAAAAXNSR0IArs4c6QAAAHxJREFUSEvtlMENgDAMA68rIQZgH4ag/cNOLAHzoDyqUgl+cR6IDuBTXNsJ8Utife6AyRm2m14FZGBxBphmUQIKkMMA5o70D5ztb3KhMZVcUS8YgM2ZMANnWIoURet6MAKrwKLjOylydue5aNKpUKRIPtdyQNeDf65fY34BzSgYGadn+YwAAAAASUVORK5CYII="/>'+
  '    </button> '+
  '    <button type="button" class="stylet-btn extra" onclick="Stylet.commit(\'${id}\', \'clearAll\')">'+
  '        <img height="24" src="data:image/png;base64, iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAYAAADgdz34AAAAAXNSR0IArs4c6QAAAVtJREFUSEvVlLkyRUEQhr/7JEioIkDAu3AjW2KJLBnKlVkiS2KL8C4ECKgiwZuoVjO3pub+PWeSW8pEp3r+6T799dKiz6fVZ//8aYAB4LsyQ1erMhgFboBpoAPsNgTZA3aAB2AOeEv1KsATMJGIDoEtJ8gBsJncPQOTpQCDwKdwdgqsZfYTYFVoh4CvaFcZGJZt8fASWAr2C2BRaPYDru6V10V56vHBXfiYFc4lylKbeghUORTCX13THBiWhYYuunJwVQUw0S2gkNidIWuXfqApg+hkxnFyXwhelYGlP9+A6LqEsZSBFW6lclWcOTPhFtlabkM4NyR2FLKjbKpdRHG35P5tP0VchsX2Tn56dleOaBh4Fw/PBS7Dsiy0I8BHtKsa2FacSh4eO7hMYljWE+1j2MJdkwowFnp/HOjZLQ4W210vYSZeU02piyxVhUs1lqutGbTKTtWy/x/gBzCOOBnlk9UvAAAAAElFTkSuQmCC"/>'+
  '    </button> '+
  ''+
  '    <!-- pen -->'+
  '    <button type="button" class="stylet-stroker-btn pen st-selected extra"  onclick="Stylet.commit(null, \'changeTool\', \'pen\')">'+
  '        <svg id="${id}-stylet-pen-ill" width="44" height="62" viewBox="0 0 44 62" fill="none" xmlns="http://www.w3.org/2000/svg">'+
  '        <path d="M14.3199 51.6339L14.3199 29.5837L11.5009 29.6036L11.5009 51.6339L14.3199 51.6339Z" fill="#C4C4C4"/>'+
  '        <path d="M25.0886 29.5228L25.0886 51.6339L17.9407 51.6339L17.9407 29.5684L25.0886 29.5228Z" fill="#C4C4C4"/>'+
  '        <path d="M32.2348 51.6339L28.7094 51.6339L28.7094 29.4997L32.2561 29.4771L32.2348 51.6339Z" fill="#C4C4C4"/>'+
  '        <path d="M32.2125 26.7552L11.4593 26.8915L17.0477 12.3448L26.4353 12.2888L32.2125 26.7552Z" fill="#C4C4C4"/>'+
  '        <path class="pentip" d="M17.7215 9.06093L20.7039 1.30059C20.8631 0.88402 21.2672 0.612216 21.7119 0.597672C22.1607 0.602612 22.5623 0.87513 22.7249 1.28769L25.8117 9.00736C25.9446 9.34403 25.906 9.72081 25.705 10.0147C25.506 10.3184 25.1715 10.4961 24.8126 10.4984L18.7435 10.5391C18.3845 10.5336 18.0517 10.364 17.843 10.0688C17.6383 9.77355 17.5969 9.39923 17.7215 9.06093Z" fill="black"/>'+
  '        </svg>'+
  '    </button>'+
  '    <!-- highlighter -->'+
  '    <button type="button" class="stylet-stroker-btn highlighter extra" onclick="Stylet.commit(null, \'changeTool\', \'highlighter\')">'+
  '        <svg id="${id}-stylet-stroker-ill" width="54" height="64" viewBox="0 0 54 64" fill="none" xmlns="http://www.w3.org/2000/svg">'+
  '            <path class="strokertip" fill-rule="evenodd" clip-rule="evenodd" d="M26.9849 16.2613L23.6287 16.2325L23.683 9.44162L30.4111 7.05725L30.3392 16.2882L26.9849 16.2613Z" fill="#F1DE35"/>'+
  '            <path class="strokertip" fill-rule="evenodd" clip-rule="evenodd" d="M32.7194 18.6415L26.964 18.6415L21.3161 18.6415L21.3161 7.75659L32.8185 3.679L32.7194 18.6415ZM26.9849 16.2613L23.6287 16.2325L23.683 9.44162L30.4111 7.05725L30.3392 16.2882L26.9849 16.2613Z" fill="#F1DE35"/>'+
  '            <path d="M13.1303 61.4217L13.1811 37.0796C18.5355 33.9138 21.8631 28.1645 21.9132 21.9084L21.9397 18.599L27.0123 18.6415L32.0867 18.6821L32.0583 21.9896C32.0347 25.1775 32.8659 28.3181 34.4669 31.0721C35.9535 33.627 38.0475 35.766 40.5483 37.3006L40.4994 61.6409L13.1303 61.4217Z" fill="#C4C4C4"/>'+
  '            <path fill-rule="evenodd" clip-rule="evenodd" d="M10.8102 63.767L10.8102 35.6695L11.4318 35.336C16.3858 32.6681 19.4879 27.5153 19.5349 21.8893L19.5803 16.2016L27.0122 16.2016L34.486 16.3211L34.4405 22.0088C34.4165 24.7704 35.139 27.4913 36.5223 29.8771C37.9168 32.2667 39.9223 34.2406 42.3256 35.5818L42.9398 35.927L42.9398 63.767L10.8102 63.767ZM13.1811 37.0797C18.5355 33.9139 21.8631 28.1645 21.9132 21.9084L21.9396 18.5989L27.0122 18.6415L32.0867 18.6822L32.0584 21.9897C32.0348 25.1776 32.8661 28.3183 34.4671 31.0723C35.9537 33.6272 38.0477 35.7663 40.5485 37.3009L40.4997 61.6412L13.1304 61.4219L13.1811 37.0797Z" fill="#C4C4C4"/>'+
  '        </svg>'+
  '            '+
  '    </button>'+
  ''+
  '    <!-- eraser -->'+
  '    <button type="button" class="stylet-stroker-btn eraser extra" onclick="Stylet.commit(null, \'changeTool\', \'eraser\')">'+
  '        <svg id="${id}-stylet-erasor-ill" width="48" height="77" viewBox="0 0 48 77" fill="none" xmlns="http://www.w3.org/2000/svg">'+
  '            <path d="M32.6049 67.584C32.8691 67.4783 33.1994 67.3197 33.2197 67.0294L33.6734 38.2764C33.6768 37.9718 33.4377 37.7211 33.1337 37.7075L15.5033 36.9408C15.1834 36.9274 14.9162 37.1818 14.9147 37.5024L14.8393 66.5656C14.8376 66.8717 14.9617 67.1648 15.1818 67.3783L31.3498 67.7161C31.9861 67.7378 32.2746 67.7161 32.6049 67.584Z" fill="#C4C4C4"/>'+
  '            <path d="M15.4932 22.6902L33.2434 23.4615C33.5617 23.4751 33.8281 23.223 33.8312 22.9041L33.9531 11.0617C33.7075 7.09528 30.0418 4.43517 25.0401 4.59377C22.2678 4.68157 19.3759 5.70405 17.4067 7.7344C15.9304 9.25654 14.9722 11.3472 14.975 14.0323L14.9545 22.1253C14.9534 22.4275 15.1917 22.6774 15.4932 22.6902Z" fill="#CE7979"/>'+
  '            <path d="M15.4622 34.682L33.1198 35.45C33.4381 35.4636 33.7046 35.2115 33.7076 34.8926L33.7969 26.2867C33.7994 25.9829 33.5612 25.7314 33.258 25.7186L15.5342 24.9477C15.2143 24.9342 14.9471 25.1886 14.9456 25.5092L14.9233 34.1173C14.9224 34.4191 15.1606 34.6691 15.4622 34.682Z" fill="#C4C4C4"/>'+
  '        </svg>'+
  '            '+
  '    </button>'+
  ''+
  '    <div class="stylet-settings extra">'+
  '        <div class="stylet-colors">'+
  '            <span class="stylet-color one" onclick="Stylet.commit(null, \'changeColor\', \'one\')"></span>'+
  '            <span class="stylet-color two" onclick="Stylet.commit(null, \'changeColor\', \'two\')"></span>'+
  '            <span class="stylet-color three" onclick="Stylet.commit(null, \'changeColor\', \'three\')"></span>'+
  '            <!-- <input class="stylet-color custom" onclick="Stylet.commit(null, \'changeColor\', \'custom\')" type="color" id="html5colorpicker" onchange="clickColor(0, -1, -1, 5)" value="#ff0000" > -->'+
  '        </div>'+
  '        <div class="stylet-sizes">'+
  '            <span class="stylet-size one" onclick="Stylet.commit(null, \'changeSize\', \'one\')"></span>'+
  '            <span class="stylet-size two" onclick="Stylet.commit(null, \'changeSize\', \'two\')"></span>'+
  '            <span class="stylet-size three" onclick="Stylet.commit(null, \'changeSize\', \'three\')"></span>'+
  '            <span class="stylet-size four" onclick="Stylet.commit(null, \'changeSize\', \'four\')"></span>'+
  '        </div>'+
  '    </div>'+
  '    <button type="button" class="stylet-btn extra minimbtn" onclick="Stylet.commit(null, \'minim\')">'+
  '        <img height="24" src="data:image/png;base64, iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAYAAADgdz34AAAABGdBTUEAALGPC/xhBQAAACBjSFJNAAB6JgAAgIQAAPoAAACA6AAAdTAAAOpgAAA6mAAAF3CculE8AAAACXBIWXMAAAsTAAALEwEAmpwYAAABWWlUWHRYTUw6Y29tLmFkb2JlLnhtcAAAAAAAPHg6eG1wbWV0YSB4bWxuczp4PSJhZG9iZTpuczptZXRhLyIgeDp4bXB0az0iWE1QIENvcmUgNS40LjAiPgogICA8cmRmOlJERiB4bWxuczpyZGY9Imh0dHA6Ly93d3cudzMub3JnLzE5OTkvMDIvMjItcmRmLXN5bnRheC1ucyMiPgogICAgICA8cmRmOkRlc2NyaXB0aW9uIHJkZjphYm91dD0iIgogICAgICAgICAgICB4bWxuczp0aWZmPSJodHRwOi8vbnMuYWRvYmUuY29tL3RpZmYvMS4wLyI+CiAgICAgICAgIDx0aWZmOk9yaWVudGF0aW9uPjE8L3RpZmY6T3JpZW50YXRpb24+CiAgICAgIDwvcmRmOkRlc2NyaXB0aW9uPgogICA8L3JkZjpSREY+CjwveDp4bXBtZXRhPgpMwidZAAAAx0lEQVRIDe2SPQ7CMAxG2wKqGGBACHGFHok7cBuuwcxROAAsjAhGftr3FWpBB5Q0C6rySS+109hx7SZJVOzA33cgbVU4xNde2dp3dRX7gKdrQNdzVrgZZBrACpZwg893uE5SB3awhwzqL9GmtAa1JpQDOXKQ0ia5nEILusCotvwWFaaq57CAI3xdcGJDmr4endcrked3dKkv0NSlDUxgBnfwnYH6PYYt6BKbAbZ3MsX8khVnBqdl608KlWbRdCU0V4yPHehFByqnbCi1zomI+AAAAABJRU5ErkJggg=="/>'+
  '    </button> '+
  '    <button type="button" class="stylet-btn expandextras" onclick="Stylet.commit(null, \'maxim\')">'+
  '        <img height="24" src="data:image/png;base64, iVBORw0KGgoAAAANSUhEUgAAABgAAAAYCAYAAADgdz34AAAAAXNSR0IArs4c6QAAAVtJREFUSEvVlLkyRUEQhr/7JEioIkDAu3AjW2KJLBnKlVkiS2KL8C4ECKgiwZuoVjO3pub+PWeSW8pEp3r+6T799dKiz6fVZ//8aYAB4LsyQ1erMhgFboBpoAPsNgTZA3aAB2AOeEv1KsATMJGIDoEtJ8gBsJncPQOTpQCDwKdwdgqsZfYTYFVoh4CvaFcZGJZt8fASWAr2C2BRaPYDru6V10V56vHBXfiYFc4lylKbeghUORTCX13THBiWhYYuunJwVQUw0S2gkNidIWuXfqApg+hkxnFyXwhelYGlP9+A6LqEsZSBFW6lclWcOTPhFtlabkM4NyR2FLKjbKpdRHG35P5tP0VchsX2Tn56dleOaBh4Fw/PBS7Dsiy0I8BHtKsa2FacSh4eO7hMYljWE+1j2MJdkwowFnp/HOjZLQ4W210vYSZeU02piyxVhUs1lqutGbTKTtWy/x/gBzCOOBnlk9UvAAAAAElFTkSuQmCC"/>'+
  '    </button> '+
  ''+
  '</div>';
  styletDOM = styletDOM.split("${id}").join(id);
  var container = document.getElementById(canvasId);
  container.style.position = "relative"
  container.innerHTML = styletDOM;
  var cntSpacing = containerSpacing();
  var colors = ['one',"two","three", "custom"];
  function setColors(obj){
    var presetLength = colors.length -1;
    for (var i = 0; i < presetLength; i++) {
      $("."+id+".stylet-toolbar .stylet-color."+colors[i]).css('backgroundColor', obj[colors[i]]);
    }
  }
  ST.tools = {
    selected: 'pen',
    pen: {
      init:function(){
        setColors(this.colors);
        setStrokeStyle(this.colors[this.colors.currentLevel]);
        ST.changeSize(Stylet.defaultLineWidth.pen);
      },
      colors: {
        currentLevel: "one",
        one:'#000000',
        two:'#0072ff',
        three:'#ff1e1e',
        custom:'#000000',
      },
      setColor:function(c){
        $("#"+id+"-stylet-pen-ill .pentip").attr("fill", c);
      },
      sizes: {
        one: 1.5,
        two: 3,
        three: 10,
        four: 15
      }
    },
    highlighter: {
      init:function(){
        setColors(this.colors);
        setStrokeStyle(this.colors[this.colors.currentLevel]);
        ST.changeSize(Stylet.defaultLineWidth.highlighter);
        // setLinejoine("")
      },
      colors: {
        currentLevel: "one",
        one:'#f1de35',
        two:'#4ce824',
        three:'#ec07b1',
        custom:'#ec6607',
      },
      setColor:function(c){
        $("#"+id+"-stylet-stroker-ill .strokertip").attr(("fill", c));
      },
      sizes: {
        one: 1,
        two: 1.5,
        three: 2,
        four: 2.5
      }
    },
    eraser: {
      init:function(){
        setStrokeStyle('#fff');
        ST.changeSize(Stylet.defaultLineWidth.eraser);
        $("."+id+".stylet-toolbar .stylet-colors").css('visibility', "hidden");
      },
      clear:function(){
        $("."+id+".stylet-toolbar .stylet-colors").css('visibility', "initial");
      },
      colors: {
        currentLevel: "one",
        one:'#fff',
        two:'#fff',
        three:'#fff',
        custom:'#fff',
      },
      setColor:function(c){},
      sizes: {
        one: 15,
        two: 30,
        three: 40,
        four: 50
      }
    }
  }
  var layers = $(container).find("canvas.stylet-canvas");
  layers.each(function(i){
    this.width = container.clientWidth ;
    this.height = container.clientHeight ;
  })
  // canvas.style.width = container.width+"px";
  var debugElem = document.createElement('p');
  container.appendChild(debugElem);
  // var highlighterContext = layers[0].getContext('2d');
  var mainContext = layers[0].getContext('2d');
  var inputContext = layers[1].getContext('2d');
  var texture = "line30";
  var lineWidth = 1.5;
  var isMousedown = false
  var points = []
  var cursor;
  // var strokeStyle = '#e7e30e79';
  var strokeStyle = '#000';
  var lineCap = 'round';
  var lineJoin = 'round';

  function setStrokeStyle(style){
    strokeStyle = style;
    mainContext.strokeStyle = style;
  }
  function setLineWidth(lw){
    lineWidth = lw;
    mainContext.lineWidth = lw;
  }

  function init(){
    if(readOnly !== undefined){
      //readOnly is a form wide variable declared in infodata.js
      hideToolbar(readOnly);
    }
    ST.tools[ST.tools.selected].init();
    drawTexture(mainContext);
  }

  var strokeHistory = [];
  var strokeFuture = [];

  /**
   * Resets the canvas
   * @return {void}
   */
  function clearLayer (layer, ctx) {
    if(!ctx){
      ctx = mainContext;
    }
    ctx.clearRect(0, 0, layer.width, layer.height)
  }

  function moveTo(x,y, ctx){
    if(!ctx){
      ctx = mainContext;
    }
    cursor = {x:x,y:y};
    ctx.moveTo(x,y);
  }

  /**
   * Remove the previous stroke from history and repaint the entire canvas based on history
   * @return {void}
   */
   ST.undo = function() {
     if(strokeHistory.length > 0){
       strokeFuture.push(strokeHistory.pop());
       draw();
       save();
     }
  }

  /**
   * Change the size of the currently selected stylus tool
   * @return {void}
   */
   ST.changeSize = function(level) {
     lineWidth = ST.tools[ST.tools.selected].sizes[level];
     Stylet.defaultLineWidth[ST.tools.selected] = level;
    $(".stylet-size").removeClass("enabled");
    $(".stylet-size."+level).addClass("enabled");
  }
  /**
   * Change the color of the currently selected stylus tool
   * @return {void}
   */
   ST.changeColor = function(level) {
    strokeStyle = ST.tools[ST.tools.selected].colors[level];
    ST.tools[ST.tools.selected].colors.currentLevel = level;
    ST.tools[ST.tools.selected].setColor(strokeStyle);
  }
  /**
   * Change the currently selected stylus tool
   * @return {void}
   */
   ST.changeTool = function(params) {
    var tool = params[0];
    var prev = $("."+id +" .stylet-stroker-btn."+SELECTIONCLASS)[0];
    prev.className = prev.className.replace(/\bst-selected\b/g, "");
    var next = $("."+id +" .stylet-stroker-btn."+tool)[0];
    next.className = next.className + " "+SELECTIONCLASS;
    setTool(tool);
  }
    
  function setTool(tool){
    if(ST.tools.selected && ST.tools[ST.tools.selected].clear){
      ST.tools[ST.tools.selected].clear();
    }
    ST.tools.selected = tool;
    ST.tools[tool].init();
  }

  /**
   * Remove the previous stroke from history and repaint the entire canvas based on history
   * @return {void}
   */
  ST.redo = function () {
     if(strokeFuture.length > 0){
       strokeHistory.push(strokeFuture.pop());
       draw();
       save();
     }
  }

  ST.clearAll = function () {
      strokeHistory.push({cmd: "clearAll"});
      initMainContext();
      save();
  }

  ST.minim = function () {
    $("."+id+".stylet-toolbar").addClass("minim");
  }

  ST.maxim = function () {
    $("."+id+".stylet-toolbar").removeClass("minim");
  }

  ST.toggleLines = function () {
    texture = texture == undefined || texture === "" ? "line30": undefined;
    draw();
    save();
  }

  function initMainContext(){
    mainContext.clearRect(0, 0, layers[0].width, layers[0].height)
    drawTexture(mainContext);
  }

  function draw(){
    initMainContext();
    if (strokeHistory.length === 0) return
    strokeHistory.map(function (action) {
      execute(action);
    });
  }


  function drawTexture(ctx){
    if(texture && allTextures[texture]){
      allTextures[texture](ctx);
    }
  }

  var allTextures = {
    line30: function(ctx){
      var width = ctx.canvas.clientWidth;
      var height = ctx.canvas.clientHeight;
      var lines = height / 30;
      ctx.strokeStyle = "#5a9ddf";
      ctx.lineWidth = "0.5";
      var end;
      for (var i = 1; i < lines; i++) {
        ctx.beginPath();
        end = i*30;
        moveTo(0, end, ctx);
        ctx.lineTo(width, end);
        ctx.stroke();
      }
    }
  }

  var allActions = {
    clearAll: function(action, ctx){
      initMainContext();
    }
  }
  var higlighterImage = "data:image/jpeg;base64, iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAYAAABzenr0AAAACXBIWXMAAC4jAAAuIwF4pT92AAACl0lEQVRYR+2XjS+QURTGf5StImVJiPJRiP7//0SiL1LpE0mENPtt52x372Tmvaw1tju3V9t57nPO85xzOjjdTxdwA7gDDAEPgfvxb793AvvAFvAReAe8BT4D34E94M9xoTpOEd//cxXoBvqBEWAceADcje8JYBP4AKwAywHGbwI4bANABnqAexH8MTABDAI3gSvAL2AdWAVexvG+AezWANAb9D8CngBTkYYEYJBv8frnwELc/ebffp+VAek1BbeAYcDXzwLTwCggMP9ukC9B/TPAYxq+Ajs1AfjyJoBMgQDexOsF4L06ABmYCwYsSJm5UAAy8LQBoEyBr/b18+fBgNo39zIwE5KUgRLA6wKA96opEIAMJACLMFOQRSgDvl4WqgNQBQlAJv5WA6mC/7cIU4YyUPqA+ZZ2TcjjXSOq4gMGSiPSCU2BNZEAtOISgG6oEbUGYDOyF2i5ev9kKEBLtjOmFdtw0oqXAI9Nyf4gA62akTKzGQ0AY2HHdkQB+F0jEoDBbMXZjLzbjGSnNQDbse3XWcBOKBC7owCyHRvsfeTe/Ht3HqgCwEAJQPpPAvAK8AjAeaA1gOY8oBc054FMgZNQpsB7tXkgi9DA2rAsWJSqIGtAFRg0i9B7lSJMFZhzAchAqkAA1kCqIAG8CDACaD0RNVVgcFWQI1kWYakCa+BcVaAaBHCpgtIJTUGqwBqoOhWXKrAPlEVYqkD7NfhiAKiqAq04VeDv43pBk4GqKnAzciMyuEUooLIXaDpuRg4iHu9VnFCK3QH7oiULorkbHsRu+CmoN//uhu6LrXZDFxp94HrMf5qRc4ESFJDfbdluPj9iJliL1+uM27G4nnk5TQDXAoC0m3uB3G4AMJhB3ZA93n/+0wCOAMxFIDDnW5ImAAAAAElFTkSuQmCC"
  var img = new Image();
  img.src = higlighterImage;
  
  function getBrushImage(stroke){
    var bc = document.createElement('canvas');
    bc.width = img.width;
    bc.height = img.height*stroke.lineWidth;
    var bctx = bc.getContext("2d")
    bctx.globalAlpha = 0.50;
    bctx.drawImage(img, 0, 0, bc.width, bc.height);

    // set composite mode
    bctx.globalCompositeOperation = "source-in";
  
    // draw color
    bctx.fillStyle = stroke.color;
    bctx.fillRect(0, 0, bc.width, bc.height);
    var res = document.createElement("img");
    res.src = bc.toDataURL("image/png");
    return res;
  }

  function execute(action, ctx){
    if(action.cmd){
      if(allActions[action.cmd]){
        allActions[action.cmd](action, ctx);
      }
    }else{
      drawStroke(action, ctx);
    }
  }

  function drawStroke(stroke, ctx){
    if(!ctx){
      ctx = mainContext;
    }
    ctx.strokeStyle = stroke.color
    ctx.lineCap = lineCap
    ctx.lineJoin = lineJoin
    ctx.lineWidth = stroke.lineWidth;
    ctx.beginPath();
    var start = stroke.path[0];
    moveTo(start.x, start.y, ctx);
    if(stroke.brush){
      strokeBrushPath(stroke, ctx);
    }else{
      strokePath(stroke, ctx);
    }
  }
  
  function strokePath(stroke, ctx){
    if(stroke.path.length >1){
      var point;
      var middle;
      var prev;
      for (var i = 1; i < stroke.path.length; i++) {
        prev = stroke.path[i - 1];
        point = stroke.path[i];
        middle =  getMiddlePoint(prev,point);
        strokeLine(prev, middle, ctx);
      }
      strokeLine(middle,point, ctx);
    }else if(stroke.path.length == 1){
      strokePoint(stroke, stroke.path[0], ctx);
    }
  }


  // function strokeLineWithVirtualDivisionPairs(a,b,div, ctx){
  //   var stroke = [a].concat(getNDivisions(a,b, div));
  //   stroke.push(b);
  //   // strokeLine(stroke[0],stroke[1]);
  //   var l = stroke.length-1;
  //   for (var i = 1; i < l; i++) {
  //     strokeLine(stroke[i - 1],stroke[i], ctx);
  //   }
  //   return stroke;
  // }
  function distanceBetween(point1, point2) {
    return Math.sqrt(Math.pow(point2.x - point1.x, 2) + Math.pow(point2.y - point1.y, 2));
  }
  function angleBetween(point1, point2) {
    return Math.atan2( point2.x - point1.x, point2.y - point1.y );
  }

  function strokeBrushPath(stroke, ctx){
    var strokeImage = getBrushImage(stroke);
    for (var i = 1; i < stroke.path.length; i++) {
      strokeHighlighter(stroke.path[i - 1], stroke.path[i], ctx, strokeImage);
    }
  }

  function strokeHighlighter(a,b, ctx, brush){
    var dist = distanceBetween(a, b);
    var angle = angleBetween(a, b);
    
    for (var i = 0; i < dist; i+=5) {
      x = a.x + (Math.sin(angle) * i) - 25;
      y = a.y + (Math.cos(angle) * i) - 25;
      ctx.drawImage(brush, x, y);
    }
  }
  function strokePoint(stroke, point, ctx){
    if(!ctx){
      ctx = mainContext;
    }
    // ctx.lineTo(a.x, a.y);
    // ctx.stroke();
    // moveTo(a.x, a.y);
    // ctx.beginPath();
    // ctx.lineTo(b.x, b.y);
    ctx.beginPath();
    moveTo(point.x, point.y, ctx);
    ctx.lineTo(point.x + 1, point.y);
    ctx.stroke();
  }

  function strokeLine(a,b, ctx){
    if(!ctx){
      ctx = mainContext;
    }
    var control = controlThreshold(a, {x:controlTangent(cursor.x, b.x, a.x),y:controlTangent(cursor.y, b.y, a.y)}, 5);
    // ctx.bezierCurveTo(control.x, control.y, control.x, control.y, b.x, b.y);
    ctx.quadraticCurveTo(control.x, control.y, b.x, b.y);
    // ctx.lineTo(a.x, a.y);
    // ctx.stroke();
    // moveTo(a.x, a.y);
    // ctx.beginPath();
    // ctx.lineTo(b.x, b.y);
    ctx.stroke();
    ctx.beginPath();
    moveTo(b.x, b.y, ctx);
  }

  function controlThreshold(tang, control, threshold){
    if(Math.abs(tang.x - control.x) > threshold || Math.abs(tang.y-control.y) > threshold){
      return control;
    }
    return tang;
  }
  function controlTangent(start, end, tangentpoint){
    return ((0.25*start) + (0.25*end) - tangentpoint)/(-0.5);
  }

  // function getNDivisions(a, b, n){
  //   var res = [];
  //   if(n % 2 === 0){
  //     var div = n/2;
  //     var middle =  getMiddlePoint(a,b);
  //     res = res.concat(getNDivisions(a,middle, div));
  //     res.push(middle);
  //     res = res.concat(getNDivisions(middle,b, div));
  //   }
  //   return res;
  // }
  function getMiddlePoint(a, b){
    return {x: (a.x + b.x) / 2, y:(a.y + b.y) / 2}
  }

  function lablePoint(point){
    mainContext.strokeStyle = 'none'
    mainContext.beginPath();
    mainContext.arc(point.x, point.y, 1.4, 0, 2 * Math.PI);
    mainContext.fillStyle = "#000";
    mainContext.fill();
    mainContext.stroke();
  }

  function lableStroke(path){
    for (var i = 0; i < path.length; i++) {
      var point = path[i];
      mainContext.beginPath();
      mainContext.arc(point.x, point.y, 1.4, 0, 2 * Math.PI);
      mainContext.fillStyle = "#e70e0e";
      mainContext.fill();
    }
    mainContext.stroke();
  }

  ST.drawOnCanvas = drawStroke;

  var palmThreshold = 30;
  var inputType;
  function addPoint(points, point, touch){
    // debugElem.innerHTML = debugElem.innerHTML +'['+point.x+','+point.y+'] '
    
    points.push(point);
  }

  function forE(list, fn){
    for (var i = 0; i < list.length; i++) {
      fn(list[i]);
    }
  }

  forE(["pointerdown"],function(ev) {
    layers[1].addEventListener(ev, function (e) {

    // debugElem.innerHTML = debugElem.innerHTML +'['+ev;
      isMousedown = true
      var p = getEventCoordinates(e);
      points = [];
      moveTo(p.x,p.y, inputContext)
      addPoint(points, p, e.touches ? e.touches[0] : null)
      // debugElem.innerHTML = debugElem.innerHTML+'<br/>';
    })
  });
  
  forE(['touchmove', 'mousemove', 'pointermove'],function(ev) {
    layers[1].addEventListener(ev, function (e) {
      if (!isMousedown) return
      e.preventDefault()

      // debugElem.innerHTML = debugElem.innerHTML +", "+ev;
      addPoint(points, getEventCoordinates(e), e.touches ? e.touches[0] : null)
      clearLayer(layers[1], inputContext);
      drawStroke(currentStroke(), inputContext)
    })
  });

  function getSelectedBrush(){
    if(ST.tools.selected == 'highlighter'){
      return 1;
    }
  }

  function currentStroke(){
    var res = {lineWidth: lineWidth, color: strokeStyle, path: smoothCompress(points)};
    if(getSelectedBrush()){
      res.brush = getSelectedBrush();
    }
    return res;
  }
  function currentStrokeUntouched(){
    var res = {lineWidth: lineWidth, color: '#e7e30e79', path: points};
    if(getSelectedBrush()){
      res.brush = getSelectedBrush();
    }
    return res;
  }

  forE(['pointerup'],function(ev) {
    layers[1].addEventListener(ev, function (e) {
      isMousedown = false

      // debugElem.innerHTML = debugElem.innerHTML +", "+ev+ "]<br/><br/>";
      if(Array.isArray(points) && points.length > 0){
        // var pointClone = [...points];
        var stroke = currentStroke();
        strokeHistory.push(stroke);
        strokeFuture.length = 0;
        clearLayer(layers[1], inputContext);
        drawStroke(stroke);
        // drawStroke(currentStrokeUntouched(), inputContext);
        save();
        // mainContext.strokeStyle = "#e70e0e42";
        // drawStroke(pointClone);
        // lableStroke(stroke.path);
      }
      points = [];
    });
  });


  function debugPoints(ps){
    moveTo(ps[0].x,ps[0].y, inputContext)
    addPoint(points, ps[0], null)
    debugElem.innerHTML = debugElem.innerHTML+'<br/>';


    for (var i = 1; i < ps.length; i++) {
      addPoint(points, ps[i], null)

      // drawOnCanvas(points);
      clearLayer(layers[1], inputContext);
      drawStroke(currentStroke(), inputContext)
    }

    var stroke = currentStroke();
    strokeHistory.push(stroke);
    strokeFuture.length = 0;
    clearLayer(layers[1], inputContext);
    drawStroke(stroke);
  }


  // debugPoints([{x:400 ,y:223},{x:178 ,y:487},{x:172 ,y:491},{x:170 ,y:493}, {x:171 ,y:491}, {x:172 ,y:489}]);
  // debugPoints([{x:616 ,y:357}, {x:197 ,y:331},{x:195 ,y:332},{x:194 ,y:333},{x:197 ,y:334}]);

  var defaultSavableStroke = {lineWidth: "", color:"", paths: []};

  function save(){
    var savableFileObj = getFileObject();
    var persistenceInput = document.getElementById(persistenceInputId);
    if(persistenceInput){
      persistenceInput.value = JSON.stringify(savableFileObj);
    }
  }

  function loadFileObject(){
    var persistenceInput = document.getElementById(persistenceInputId);
    if(persistenceInput && !!persistenceInput.value){
       var savedFile = JSON.parse(persistenceInput.value);
       loadFile(savedFile);
    }
  }
  typeof FormEvent !== "undefined" && FormEvent.on('variablesLoaded', function(loadedVariables){
    loadFileObject();
  });
  typeof FormEvent !== "undefined" && FormEvent.on('onReadOnlyChange', function(params){
    hideToolbar(params.readOnly);
  });
  
  function hideToolbar(hide){
    $("."+id+".stylet-toolbar").css('visibility', hide?"hidden":'initial');
  }

  // function stickyToolbar(){
  //   var observer = new window.IntersectionObserver(([entry]) => {
  //     var isBottomVisible = container.getBoundingClientRect().bottom <= (window.innerHeight || document.documentElement.clientHeight);
  //     if (entry.isIntersecting  && !isBottomVisible) {
  //       $("."+id+".stylet-toolbar").addClass("stylet-sticky");
  //       console.log('ENTER'+ id)
  //       return
  //     }else{
  //       $("."+id+".stylet-toolbar").removeClass("stylet-sticky");
  //     }
  //     console.log('LEAVE' + id)
  //   });
    
  //   observer.observe(container);
  // }
  // stickyToolbar();

  var stickyThreshold = [];
  for (var i = 1; i < 51; i++) {
    stickyThreshold.push(parseFloat((i*0.02).toFixed(2)));
  }

  function stickyToolbar(){
    if(typeof window.IntersectionObserver != "undefined"){
      var observer = new window.IntersectionObserver(function(obs) {
        var entry = obs[0];
        var isBottomVisible = entry.boundingClientRect.bottom <= entry.intersectionRect.bottom;
        if (entry.isIntersecting  && !isBottomVisible) {
          $("."+id+".stylet-toolbar").addClass("stylet-sticky");
        }else{
          $("."+id+".stylet-toolbar").removeClass("stylet-sticky");
        }
      }, {threshold: stickyThreshold});
      
      observer.observe(container);
    }
  }
  stickyToolbar();

  function loadFile(file){
    strokeHistory.length = 0;
    if(file){
      texture = file.texture;
    }

    var drawing, path, s, i, d;
    for (i = 0; i < file.drawings.length; i++) {
      drawing = file.drawings[i];
      if(drawing.cmd){
        strokeHistory.push(drawing);
      }else if(drawing.strokes){
        for (d = 0; d < drawing.strokes.paths.length; d++) {
          path = drawing.strokes.paths[d];
          s = {
            lineWidth: drawing.strokes.lineWidth,
            color: drawing.strokes.color,
            path: path.x.map(function(e,yI){
                return {x:e, y: path.y[yI]};
            })
          };
          if(drawing.strokes.brush){
            s.brush = drawing.strokes.brush;
          }
          strokeHistory.push(s);
        }
      }
    }
    //initial loading bug for highligter requires doing the draw twice. Otherwise the highliter does not draw on load.
    setTimeout(function(){
      draw();
      setTimeout(draw, 20);
    }, 50);
  }

  function getFileObject(){
    var res ={drawings:[]};
    if(texture){
      res.texture = texture;
    }

    var strokes = JSON.parse(JSON.stringify(defaultSavableStroke));

    function commitStroke(){
      if(strokes.paths.length > 0){
        res.drawings.push({strokes:strokes})
      }
    }

    function addStroke(stroke){
      strokes.paths.push(stroke.path.reduce(function(acc, point){
        acc.x.push(point.x);
        acc.y.push(point.y);
        return acc;
      }, {x:[], y:[]}));
    }

    for (var i = 0; i < strokeHistory.length; i++) {
      var stroke = strokeHistory[i];
      if(isSameTool(strokes, stroke)){
        addStroke(stroke)
      }else if(stroke.cmd){
        commitStroke();
        res.drawings.push(stroke);
        strokes = JSON.parse(JSON.stringify(defaultSavableStroke));
      }else{
        commitStroke();
        strokes = JSON.parse(JSON.stringify(defaultSavableStroke));
        strokes.lineWidth = stroke.lineWidth;
        strokes.color = stroke.color;
        if(stroke.brush){
          strokes.brush = stroke.brush;
        }
        addStroke(stroke)
      }
    }
    commitStroke();
    return res;
  }


  function isSameTool(tool, stroke){
    return tool.lineWidth === stroke.lineWidth && tool.color === stroke.color;
  }

  function smoothCompress(initial){
    // var alerttxt = 'Before smoothing <br/>================ <br/> <br/>' +JSON.stringify(initial)+'<br/>================ <br/> <br/>';
    // if(initial.length > 3){
      var res = JSON.parse(JSON.stringify(ST.simplify(initial, 1.5)));
      // alerttxt += 'After smoothing <br/>================ <br/> <br/>' +JSON.stringify(res)+'<br/>';
      // debugElem.innerHTML = alerttxt;
      return res;
    // }
    // return initial;
  }
  function containerSpacing(){
    var style = container.currentStyle || window.getComputedStyle(container);
    return {
      top:  parseFloat(style.borderTopWidth),
      left:  parseFloat(style.borderLeftWidth),
    };
  }
  function getEventCoordinates(e){
      var x, y
      if (e.touches && e.touches[0] && typeof e.touches[0]["force"] !== "undefined") {
        x = e.touches[0].pageX - (cntSpacing.left+container.offsetLeft);
        y = e.touches[0].pageY - (cntSpacing.top+container.offsetTop)  ;
      } else {
        x = e.pageX - (cntSpacing.left+container.offsetLeft);
        y = e.pageY - (cntSpacing.top+container.offsetTop);
      }
      return {x:x, y:y};
  }

  init()
}

Stylet.commit = function(id, action){
  if(id){
    Stylet.all[id][action](Array.prototype.slice.call(arguments, 2));
  }else if(action){
    for (var key in Stylet.all) {
      Stylet.all[key][action](Array.prototype.slice.call(arguments, 2));
    }
  }
  return false;
}
Stylet.all = {};
Stylet.defaultLineWidth = {pen: "one", highlighter:"one", eraser:"one" };

Stylet.skip = function(e){
  return false;
};

(function (style) {
  var css = document.createElement('style');
  css.type = 'text/css';

  if (css.styleSheet) 
      css.styleSheet.cssText = style;
  else 
      css.appendChild(document.createTextNode(style));
    
  /* Append style to the head element */
  document.getElementsByTagName("head")[0].appendChild(css);
})('.stylet-canvas{'+
'    top: 0;'+
'    left: 0;'+
'    cursor: url(/tools/stylet_cur.png), crosshair;'+
'    position: absolute;'+
'}'+
'.stylet-sizes{'+
'    display: flex;'+
'}'+
'.stylet-sticky{'+
'    position: fixed!important;'+
'}'+
'.stylet-size.one::after{ height: 0.2em; width: 0.2em;}'+
'.stylet-size.two::after{ height: 0.5em; width: 0.5em;}'+
'.stylet-size.three::after{ height: 0.8em; width: 0.8em;}'+
'.stylet-size.four::after{ height: 1em; width: 1em;}'+
'.stylet-size::after{'+
'    content: \'\';'+
'    background-color: #000;'+
'    display: inline-block;'+
'    border-radius: 50%;'+
'    align-self: center;'+
'}'+
'.stylet-size{'+
'    display: flex;'+
'    flex: 1;'+
'    cursor:default;'+
'    justify-content: center;'+
'}'+
'.stylet-color.one{ background-color: black;}'+
'.stylet-color.two{ background-color: #0072ff;}'+
'.stylet-color.three{ background-color: red;}'+
'.stylet-color.custom{ '+
'    padding: 0;'+
'    border: none;'+
'    border-radius: 0;'+
'    height: 20px;'+
'    width: 20px;'+
'    vertical-align: top;'+
'}'+
'.stylet-color{'+
'    display: inline-block;'+
'    width: 1.5em;'+
'    height: 1.5em;'+
'    border-radius: 50%;'+
'}'+
'button.stylet-stroker-btn svg {'+
'    margin-top: 20px;'+
'}'+
'.stylet-toolbar {'+
'    height: 45px;'+
'    width:34em;'+
'    overflow: hidden;'+
'    border-radius: 20px;'+
'    box-shadow: 0px 1px 15px 0px rgb(0 0 0 / 29%);'+
'    display: flex;'+
'    padding: 5px 15px;'+
'    left: calc(50% - 16em);'+
'}'+
'.stylet-stroker-btn.st-selected svg{'+
'    margin-top: 0px;'+
'}'+
'.stylet-stroker-btn{'+
'    border: none;'+
'    background-color: inherit;'+
'}'+
'.stylet-btn:hover, .stylet-size:hover, .stylet-size.enabled{'+
'    background-color: #cecece;'+
'}'+
'.stylet-stroker-btn:hover svg{'+
'    margin-top: 0px;'+
'}'+
'.stylet-toolbar.minim .extra{'+
'    display:none;'+
'}'+
'.stylet-toolbar.minim{'+
'    width:3em;'+
'    left:2em;'+
'}'+
'.stylet-toolbar .expandextras{'+
'    display:none;'+
'}'+
'.expandextras img{'+
'    transform: rotate(45deg);'+
'}'+
'.stylet-toolbar.minim .expandextras{'+
'    display:initial;'+
'}'+
'.stylet-btn.minimbtn{'+
'    margin-left:auto;'+
'}'+
'@media print {'+
' .st-no-print { display: none; }'+
'}'+
'.stylet-btn{'+
'    padding: 0.7em;'+
'    border: none;'+
'    background-color: inherit;'+
'    margin-right: 5px;'+
'}');


/*
 (c) 2017, Vladimir Agafonkin
 Simplify.js, a high-performance JS polyline simplification library
 mourner.github.io/simplify-js
*/

/**
 * Albert B. Modifications:
 * 1. Exporting removed since it's only use
 */

(function () { 'use strict';

// to suit your point format, run search/replace for '.x' and '.y';
// for 3D version, see 3d branch (configurability would draw significant performance overhead)

// square distance between 2 points
function getSqDist(p1, p2) {

    var dx = p1.x - p2.x,
        dy = p1.y - p2.y;

    return dx * dx + dy * dy;
}

// square distance from a point to a segment
function getSqSegDist(p, p1, p2) {

    var x = p1.x,
        y = p1.y,
        dx = p2.x - x,
        dy = p2.y - y;

    if (dx !== 0 || dy !== 0) {

        var t = ((p.x - x) * dx + (p.y - y) * dy) / (dx * dx + dy * dy);

        if (t > 1) {
            x = p2.x;
            y = p2.y;

        } else if (t > 0) {
            x += dx * t;
            y += dy * t;
        }
    }

    dx = p.x - x;
    dy = p.y - y;

    return dx * dx + dy * dy;
}
// rest of the code doesn't care about point format

// basic distance-based simplification
function simplifyRadialDist(points, sqTolerance) {

    var prevPoint = points[0],
        newPoints = [prevPoint],
        point;

    for (var i = 1, len = points.length; i < len; i++) {
        point = points[i];

        if (getSqDist(point, prevPoint) > sqTolerance) {
            newPoints.push(point);
            prevPoint = point;
        }
    }

    if (prevPoint !== point) newPoints.push(point);

    return newPoints;
}

function simplifyDPStep(points, first, last, sqTolerance, simplified) {
    var maxSqDist = sqTolerance,
        index;

    for (var i = first + 1; i < last; i++) {
        var sqDist = getSqSegDist(points[i], points[first], points[last]);

        if (sqDist > maxSqDist) {
            index = i;
            maxSqDist = sqDist;
        }
    }

    if (maxSqDist > sqTolerance) {
        if (index - first > 1) simplifyDPStep(points, first, index, sqTolerance, simplified);
        simplified.push(points[index]);
        if (last - index > 1) simplifyDPStep(points, index, last, sqTolerance, simplified);
    }
}

// simplification using Ramer-Douglas-Peucker algorithm
function simplifyDouglasPeucker(points, sqTolerance) {
    var last = points.length - 1;

    var simplified = [points[0]];
    simplifyDPStep(points, 0, last, sqTolerance, simplified);
    simplified.push(points[last]);

    return simplified;
}

// both algorithms combined for awesome performance
function simplify(points, tolerance, highestQuality) {

    if (points.length <= 2) return points;

    var sqTolerance = tolerance !== undefined ? tolerance * tolerance : 1;

    points = highestQuality ? points : simplifyRadialDist(points, sqTolerance);
    points = simplifyDouglasPeucker(points, sqTolerance);

    return points;
}

// export as AMD module / Node module / browser or worker variable
Stylet.prototype.simplify = simplify;

})();