/*******************************************************************************
 * Copyright (c) Steeve St-Laurent - www.infodata.ca - sstlaurent@infodata.ca
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    sstlaurent@infodata.ca - initial API
 *******************************************************************************/

package org.eclipse.nebula.widgets.calendarcombo;

import java.util.Locale;

/**
 * a factory used to create "date implementation independant" components
 */
public interface IDateFactory {

	/**
	 * @return never null
	 */
	Locale getLocale();

	/**
	 * create a new date instance
	 * @param day start from 1 to month end
	 * @param month start from 1 to 12
	 * @param year
	 * @return never null
	 * @throws NullPointerException if date is invalid
	 */
	Object newDate(int day, int month, int year);
	/**
	 * clear date instance
	 * @param date cant be null
	 * @return cleared date, never null, may or may not return same instance as param date
	 * @throws NullPointerException if date is null
	 */
	Object clear(Object date);
	/**
	 * create new today date instance
	 * @return today date, never null
	 */
	Object today();
	/**
	 * add x day to date instance
	 * @param date cant be null
	 * @param day
	 * @return modifed date, never null, may or may not return same instance as param date
	 * @throws NullPointerException if date is null
	 */
	Object plusDay(Object date, int day);
	/**
	 * add x month to date instance
	 * @param date cant be null
	 * @param month
	 * @return modifed date, never null, may or may not return same instance as param date
	 * @throws NullPointerException if date is null
	 */
	Object plusMonth(Object date, int month);
	/**
	 * add x year to date instance
	 * @param date cant be null
	 * @param year
	 * @return modifed date, never null, may or may not return same instance as param date
	 * @throws NullPointerException if date is null
	 */
	Object plusYear(Object date, int year);
	/**
	 * set date instance day to x
	 * @param date cant be null
	 * @param day, start from 1 to monthEnd
	 * @return modifed date, never null, may or may not return same instance as param date
	 * @throws NullPointerException if date is null or day is invalid
	 */
	Object withDay(Object date, int day);
	/**
	 * set date instance year to x
	 * @param date cant be null
	 * @param year
	 * @return modifed date, never null, may or may not return same instance as param date
	 * @throws NullPointerException if date is null
	 */
	Object withYear(Object date, int year);
	/**
	 * set date instance month to x
	 * @param date cant be null
	 * @param month, start from 1 to 12
	 * @return modifed date, never null, may or may not return same instance as param date
	 * @throws NullPointerException if date is null or month is invalid
	 */
	Object withMonth(Object date, int month);

	/**
	 * set date instance dayOfWeek to x
	 * @param date cant be null
	 * @param dayOfWeek, start from 1 to 7, sunday to saturday
	 * @return modifed date, never null, may or may not return same instance as param date
	 * @throws NullPointerException if date is null or month is invalid
	 */
	Object withDayOfWeek(Object date, int dayOfWeek);

	/**
	 * @param date cant be null
	 * @return date day, start from 1 to 31
	 * @throws NullPointerException if date is null
	 */
	int getDay(Object date);
	/**
	 * @param date cant be null
	 * @return date month, start from 1 to 12
	 * @throws NullPointerException if date is null
	 */
	int getMonth(Object date);
	/**
	 * @param date cant be null
	 * @return date year
	 * @throws NullPointerException if date is null
	 */
	int getYear(Object date);
	/**
	 * @param date cant be null
	 * @return date day of week, start from 1 to 7, sunday to saturday
	 * @throws NullPointerException if date is null
	 */
	int getDayOfWeek(Object date);

	/**
	 * @param date1 can be null
	 * @param date2 can be null
	 * @return true if date1 equals date2
	 */
	boolean equals(Object date1, Object date2);

	/**
	 * @param date cant be null
	 * @return a clone of date
	 * @throws NullPointerException if date is null
	 */
	Object clone(Object date);

	/**
	 * clone date1 instance to date2 instance
	 * @param date1 cant be null
	 * @param date2 cant be null
	 * @return date2
	 * @throws NullPointerException if date is null
	 */
	Object clone(Object date1, Object date2);

    /**
     * Gets what the first day of the week is; e.g., <code>SUNDAY</code> in the U.S.,
     * <code>MONDAY</code> in France.
     *
     * @return the first day of the week. 1 to 7, sunday to saturday
     */
	int getFirstDayOfWeek();

}
