/*******************************************************************************
 * Copyright (c) Steeve St-Laurent - www.infodata.ca - sstlaurent@infodata.ca
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    sstlaurent@infodata.ca - initial implementations
 *******************************************************************************/

package org.eclipse.nebula.widgets.calendarcombo;

import java.util.Calendar;
import java.util.Locale;

/**
 * a factory that work on calendar
 */
public class CalendarFactory implements IDateFactory {

	private final Locale locale;
	private final int firstDayOfWeek;

	/**
	 * @param new factory with locale set to {@link Locale#getDefault()}
	 */
	public CalendarFactory() {
		this(null);
	}

	/**
	 * @param locale can be null, default to {@link Locale#getDefault()}
	 */
	public CalendarFactory(Locale locale) {
		if (locale == null) {
			locale = Locale.getDefault();
		}
		this.locale = locale;
		this.firstDayOfWeek = ((Calendar)today()).getFirstDayOfWeek();
	}

	public Object plusDay(Object date, int day) {
		Calendar c = ((Calendar)date);
		c.add(Calendar.DATE, day);
		return c;
	}

	public Object plusMonth(Object date, int month) {
		Calendar c = ((Calendar)date);
		c.add(Calendar.MONTH, month);
		return c;
	}

	public Object plusYear(Object date, int year) {
		Calendar c = ((Calendar)date);
		c.add(Calendar.YEAR, year);
		return c;
	}

	public Object clear(Object date) {
		Calendar c = ((Calendar)date);
		c.clear();
		return c;
	}

	public boolean equals(Object date1, Object date2) {
		if (date1 == date2) {
			return true;
		}
		if (date1 == null || date2 == null) {
			return false;
		}
		return DateHelper.sameDate((Calendar)date1, (Calendar)date2);
	}

	public int getDay(Object date) {
		Calendar c = ((Calendar)date);
		return c.get(Calendar.DATE);
	}

	public int getDayOfWeek(Object date) {
		Calendar c = ((Calendar)date);
		return c.get(Calendar.DAY_OF_WEEK);
	}

	public int getMonth(Object date) {
		Calendar c = ((Calendar)date);
		return c.get(Calendar.MONTH) + 1;
	}

	public int getYear(Object date) {
		Calendar c = ((Calendar)date);
		return c.get(Calendar.YEAR);
	}

	public Object newDate(int day, int month, int year) {
		Calendar c = Calendar.getInstance(this.locale);
		c.set(Calendar.YEAR, year);
		c.set(Calendar.MONTH, month-1);
		c.set(Calendar.DATE, day);
		return toMidnight(c);
	}

	private static Calendar toMidnight(Calendar c){
		c.set(Calendar.HOUR, 0);
		c.set(Calendar.HOUR_OF_DAY, 0);
		c.set(Calendar.MINUTE, 0);
		c.set(Calendar.SECOND, 0);
		c.set(Calendar.MILLISECOND, 0);
		return c;
	}

	public Object today() {
		return toMidnight(Calendar.getInstance(this.locale));
	}

	public Object withDay(Object date, int day) {
		Calendar c = ((Calendar)date);
		c.set(Calendar.DAY_OF_MONTH, day);
		return c;
	}

	public Object withMonth(Object date, int month) {
		Calendar c = ((Calendar)date);
		c.set(Calendar.MONTH, month-1);
		return c;
	}

	public Object withYear(Object date, int year) {
		Calendar c = ((Calendar)date);
		c.set(Calendar.YEAR, year);
		return c;
	}

	public Object withDayOfWeek(Object date, int dayOfWeek) {
		Calendar c = ((Calendar)date);
		c.set(Calendar.DAY_OF_WEEK, dayOfWeek);
		return c;
	}

	public Object clone(Object date) {
		Calendar c = ((Calendar)date);
		return c.clone();
	}

	public int getFirstDayOfWeek() {
		return this.firstDayOfWeek;
	}

	public Object clone(Object date1, Object date2) {
		Calendar c1 = ((Calendar)date1);
		Calendar c2 = ((Calendar)date2);
		c2.set(Calendar.YEAR, c1.get(Calendar.YEAR));
		c2.set(Calendar.MONTH, c1.get(Calendar.MONTH));
		c2.set(Calendar.DATE, c1.get(Calendar.DATE));
		return toMidnight(c2);
	}

	public Locale getLocale() {
		return this.locale;
	}
}
