/*******************************************************************************
 * Copyright (c) Emil Crumhorn - Hexapixel.com - emil.crumhorn@gmail.com
 * Copyright (c) Steeve St-Laurent - www.infodata.ca - sstlaurent@infodata.ca
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    emil.crumhorn@gmail.com - initial API and implementation of CalendarComposite
 *    sstlaurent@infodata.ca - copy and epuration of CalendarComposite
 *******************************************************************************/

package org.eclipse.nebula.widgets.calendarcombo;


import java.text.DateFormatSymbols;
import java.util.ArrayList;
import java.util.Locale;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.MouseMoveListener;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Layout;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;

/**
 * this class does not support {@link ISettings#showMonthPickerOnMonthNameMousePress()} because if does not support month picker<br>
 * this class does not support {@link IColorManager#getDisabledDayForegroundColor()} because it does not support disallowed dates<br>
 * this class is really independant from {@link CalendarCombo}<br>
 * this class allow user to use its own date class via {@link IDateFactory} interface
 */
public class CalendarComposite2 extends Canvas implements MouseListener, MouseMoveListener {

	public static interface IDateListener {
		/**
		 * @see IDateFactory for date type
		 * @param date
		 */
		void dateChanged(Object date);
	}

	private final int					ARROW_LEFT					= 1;

	private final int					ARROW_RIGHT					= 2;

	private Button						mButtonToday;

	private Button						mButtonNone;

	private Rectangle					mLeftArrowBounds;

	private Rectangle					mRightArrowBounds;

	private Object					mCalendar;

	private Object					mToday;

	private int							mDatesTopY					= 0;

	private int							mDayXs[]					= new int[7];

	private CalDay[]					mDays						= new CalDay[7 * 6];

	private Object					mSelectedDay;

	private static DateFormatSymbols	mDFS;

	private String						mMonths[];

	private static String[]				mDayTitles					= null;

	private ArrayList mListeners;

	private boolean						mNoDayClicked;

	// this thread deals with holding down the mouse over the arrows to jump
	// months quickly

	// milliseconds
	private static final int			ARROW_SLOW_TIME				= 300;

	private static final int			ARROW_FAST_TIME				= 120;

	// after how many iterations do we switch speed? (after how many months
	// flipped-by)
	private static final int			ARROW_SPEED_SWITCH_COUNT	= 10;

	// default speed
	private int							mArrowSleepTime				= ARROW_SLOW_TIME;

	// various variables used to keep track
	private int							mArrowIterations			= 0;

	private Thread						mArrowThread;

	private boolean						mArrowRun;

	private boolean						mArrowPause;

	private int							mArrowThreadDirection		= 0;

	private final IColorManager				mColorManager;

	private final ISettings					mSettings;

	private final IDateFactory				mDateFactory;

	public CalendarComposite2(Composite parent, IColorManager colorManager, ISettings settings, IDateFactory dateFactory) {
		super(parent, SWT.NO_BACKGROUND | SWT.DOUBLE_BUFFERED);
		this.mColorManager = colorManager;
		this.mSettings = settings;
		this.mDateFactory = dateFactory == null ? new CalendarFactory(this.mSettings == null ? null : this.mSettings.getLocale()) : dateFactory;

		init();

		build();
	}

	private void init() {
		mDFS = new DateFormatSymbols(this.mDateFactory.getLocale());
		this.mMonths = mDFS.getMonths();

		if (this.mCalendar == null) {
			this.mCalendar = this.mDateFactory.today();
		}
		if (this.mToday == null) {
			this.mToday = this.mDateFactory.today();
		}

		String[] weekdays = mDFS.getWeekdays();
		mDayTitles = new String[weekdays.length];
		for (int i = 0; i < weekdays.length; i++) {
			String weekday = weekdays[i];
			if (weekday.length() > 0) {
				mDayTitles[i] = weekday.substring(0, 1).toUpperCase();
			}
		}

		//will be reused in draw days methods
		for (int i = 0; i < 42; i++) {
			this.mDays[i] = new CalDay(i, this.mDateFactory.today(), new Rectangle(0,0,0,0));
		}

	}

	private void build() {
		// button height & width and spacers
		int bheight = this.mSettings.getButtonHeight();
		int bwidth = this.mSettings.getButtonWidth();
		int buttonStyle = SWT.PUSH;
		this.mListeners = new ArrayList();

		// Mac buttons need different flag to look normal
		if (CalendarCombo.OS_CARBON) {
			bheight = this.mSettings.getCarbonButtonHeight();
			bwidth = this.mSettings.getButtonWidthCarbon();
			buttonStyle = SWT.FLAT;
		}

		setLayout(new ButtonSectionLayout());

		addPaintListener(new PaintListener() {
			public void paintControl(PaintEvent event) {
				paint(event);
			}
		});

		addMouseListener(this);
		addMouseMoveListener(this);
		addKeyListener(new KeyListener() {
			public void keyReleased(KeyEvent e) {
			}
			public void keyPressed(KeyEvent e) {
				CalendarComposite2.this.keyPressed(e.keyCode, e.stateMask);
			}
		});

		this.mButtonToday = new Button(this, buttonStyle | SWT.NO_FOCUS);
		this.mButtonToday.setText(this.mSettings.getTodayText());
		this.mButtonToday.setLayoutData(new GridData(bwidth, bheight));
		this.mButtonToday.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event event) {
				clickedTodayButton();
			}
		});

		this.mButtonNone = new Button(this, buttonStyle | SWT.NO_FOCUS);
		this.mButtonNone.setText(this.mSettings.getNoneText());
		this.mButtonNone.setLayoutData(new GridData(bwidth, bheight));
		this.mButtonNone.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event event) {
				clickedNoneButton();
			}
		});
	}

	private void clickedTodayButton() {
		setReferenceDate(this.mDateFactory.today());
		this.mSelectedDay = this.mDateFactory.today();
		notifyListeners();
	}

	private void clickedNoneButton() {
		selectDate(null);
		notifyListeners();
	}

	private void paint(PaintEvent event) {
		GC gc = event.gc;
		drawChartOntoGC(gc);
	}

	private void drawChartOntoGC(GC gc) {
		Rectangle bounds = super.getBounds();
		gc.setBackground(this.mColorManager.getCalendarBackgroundColor());
		gc.fillRectangle(bounds);

		Font used = null;
		if (CalendarCombo.OS_CARBON) {
			used = this.mSettings.getCarbonDrawFont();
			if (used != null) {
				gc.setFont(used);
			}
		}

		// header
		drawHeader(gc);
		// day titles
		drawTitleDays(gc);
		// days
		drawDays(gc);
		// 1 pixel border
		drawBorder(gc);

		gc.dispose();
		if (used != null) {
			used.dispose();
		}
	}

	public void setReferenceDate(Object date) {
		this.mCalendar = date == null ? this.mDateFactory.today() : this.mDateFactory.clone(date);
		redraw();
	}

	public void selectDate(Object date){
		this.mSelectedDay = date == null ? null : this.mDateFactory.clone(date);
		redraw();
	}

	public void nextMonth() {
		this.mSelectedDay = null;
		this.mCalendar = this.mDateFactory.plusMonth(this.mCalendar, 1);
		this.mCalendar = this.mDateFactory.withDay(this.mCalendar, 1);
		redraw();
	}

	public void prevMonth() {
		this.mSelectedDay = null;
		this.mCalendar = this.mDateFactory.plusMonth(this.mCalendar, -1);
		this.mCalendar = this.mDateFactory.withDay(this.mCalendar, 1);
		redraw();
	}

	public void goToToday() {
		this.mSelectedDay = null;
		this.mCalendar = this.mDateFactory.today();
		redraw();
	}

	// draws 1 pixel border around entire calendar
	private void drawBorder(GC gc) {
		Rectangle bounds = super.getBounds();
		gc.setForeground(this.mColorManager.getCalendarBorderColor());
		gc.drawRectangle(bounds.x, bounds.y, bounds.width - 1, bounds.height - 1);
	}

	private void drawHeader(GC gc) {
		Rectangle bounds = super.getBounds();
		Rectangle bgRect = new Rectangle(bounds.x + this.mSettings.getHeaderLeftMargin(), bounds.y + this.mSettings.getHeaderTopMargin(), this.mSettings.getCalendarWidth() - 13, bounds.y
				+ this.mSettings.getHeaderHeight());
		gc.setBackground(this.mColorManager.getCalendarHeaderColor());
		gc.fillRectangle(bgRect);
		drawArrow(gc, bounds.x + this.mSettings.getHeaderLeftMargin() + this.mSettings.getArrowLeftSpacing() + 1, bounds.y + this.mSettings.getHeaderTopMargin() + this.mSettings.getArrowTopSpacing()
				+ 4, this.ARROW_LEFT);
		drawArrow(gc, bounds.x + this.mSettings.getCalendarWidth() - 13 - this.mSettings.getHeaderRightMargin(), bounds.y + this.mSettings.getHeaderTopMargin() + this.mSettings.getArrowTopSpacing()
				+ 4, this.ARROW_RIGHT);

		String toDraw = this.mMonths[this.mDateFactory.getMonth(this.mCalendar)-1] + " " + this.mDateFactory.getYear(this.mCalendar);
		Point strWidth = gc.stringExtent(toDraw);

		int avail = this.mSettings.getCalendarWidth() - 13 - strWidth.x;
		avail /= 2;

		gc.drawString(toDraw, bounds.x + this.mSettings.getHeaderLeftMargin() + avail, bounds.y + this.mSettings.getHeaderTopMargin() + 1, true);
	}

	private void drawTitleDays(GC gc) {
		// fetch the first day of the week, and draw starting on that day
		int fdow = this.mDateFactory.getFirstDayOfWeek();

		Rectangle bounds = super.getBounds();
		int xStart = this.mSettings.getDatesLeftMargin() + 5;
		int yStart = bounds.y + this.mSettings.getHeaderTopMargin() + this.mSettings.getHeaderHeight() + 1;

		int spacer = 0;
		int letterHeight = 0;

		for (int i = 0; i < 7; i++) {
			Point strWidth = gc.stringExtent(mDayTitles[fdow]);

			int x = xStart + this.mSettings.getOneDateBoxSize() + spacer - strWidth.x;
			// don't add the string width, as our string width later when
			// drawing days will differ
			this.mDayXs[i] = xStart + this.mSettings.getOneDateBoxSize() + spacer;

			gc.drawString(mDayTitles[fdow], x, yStart, true);

			letterHeight = strWidth.y;
			spacer += this.mSettings.getOneDateBoxSize() + this.mSettings.getBoxSpacer();

			fdow++;
			if (fdow > 7) {
				fdow = 1;
			}
		}

		int lineStart = yStart + 1 + letterHeight;
		gc.setForeground(this.mColorManager.getLineColor());
		gc.drawLine(this.mSettings.getDatesLeftMargin() + 1, lineStart, bounds.width - this.mSettings.getDatesRightMargin() - 3, lineStart);

		this.mDatesTopY = lineStart + 3;
	}

	private void drawDays(GC gc) {
		gc.setBackground(this.mColorManager.getCalendarBackgroundColor());
		gc.setForeground(this.mColorManager.getTextColor());

		Rectangle bounds = super.getBounds();
		int spacer = 0;

//		debug("mCalendar", this.mCalendar);
		Object start = this.mDateFactory.clone(this.mCalendar);
//		debug("start", start);
		start = this.mDateFactory.withDay(start, 1);
//		debug("start", start);
		start = this.mDateFactory.withDayOfWeek(start, 1);
//		debug("start", start);

		Object date = start;

		final int monthToShow = this.mDateFactory.getMonth(this.mCalendar);

		int col = 0;
		int colCount = 0;

		String typicalLetter = "8";
		int strHeight = gc.stringExtent(typicalLetter).y;

		gc.setForeground(this.mColorManager.getLineColor());

		int lastY = 0;

		for (int y = 0; y < 42; y++) {

//			debug(date);

			// new row
			if (y % 7 == 0 && y != 0) {
				spacer += strHeight + 2;
			}

			if (colCount > 6) {
				colCount = 0;
				col = 0;
			}

			if (this.mDateFactory.getMonth(date) == monthToShow) {
				gc.setForeground(this.mColorManager.getTextColor());
			} else {
				gc.setForeground(this.mColorManager.getPreviousAndNextMonthForegroundColor());
			}

			String dateStr = String.valueOf(this.mDateFactory.getDay(date));
			Point width = gc.stringExtent(dateStr);

			if (this.mDateFactory.equals(this.mSelectedDay, date)) {
				gc.setBackground(this.mColorManager.getSelectedDayColor());
				gc.fillRectangle(this.mDayXs[col] - this.mSettings.getOneDateBoxSize() - 4, this.mDatesTopY + spacer - 1, this.mSettings.getOneDateBoxSize() + 5, 14);
				gc.setBackground(this.mColorManager.getCalendarBackgroundColor());
			}

			this.mDays[y].bounds.x = this.mDayXs[col] - this.mSettings.getOneDateBoxSize() - 4;
			this.mDays[y].bounds.y = this.mDatesTopY + spacer - 1;
			this.mDays[y].bounds.width = this.mSettings.getOneDateBoxSize() + 5;
			this.mDays[y].bounds.height = 14;
			this.mDays[y].number = y;
			this.mDays[y].date = this.mDateFactory.clone(date, this.mDays[y].date);

			gc.drawString(dateStr, this.mDayXs[col] - width.x, this.mDatesTopY + spacer, true);

			if (this.mDateFactory.equals(this.mToday, date)) {
				Color old = gc.getForeground();
				gc.setForeground(this.mColorManager.getSelectedDayBorderColor());
				gc.drawRectangle(this.mDayXs[col] - this.mSettings.getOneDateBoxSize() - 4, this.mDatesTopY + spacer - 1, this.mSettings.getOneDateBoxSize() + 5, 14);
				gc.setForeground(old);
			}

			date = this.mDateFactory.plusDay(date, 1);
			col++;
			colCount++;
			lastY = this.mDatesTopY + spacer;
		}

		lastY += strHeight + 1;

		gc.setForeground(this.mColorManager.getLineColor());
		gc.drawLine(this.mSettings.getDatesLeftMargin() + 1, lastY, bounds.width - this.mSettings.getDatesRightMargin() - 3, lastY);
	}

	private void debug(Object date) {
		System.out.println(this.mDateFactory.getDay(date) + "-"+this.mDateFactory.getMonth(date) +"-"+this.mDateFactory.getYear(date));
	}
	private void debug(String text, Object date) {
		System.out.println(text + " " + this.mDateFactory.getDay(date) + "-"+this.mDateFactory.getMonth(date) +"-"+this.mDateFactory.getYear(date));
	}

	private void drawArrow(GC gc, int x, int y, int style) {
		gc.setForeground(this.mColorManager.getArrowColor());
		switch (style) {
		case ARROW_RIGHT:
			gc.drawLine(x, y - 4, x, y + 4);
			gc.drawLine(x + 1, y - 3, x + 1, y + 3);
			gc.drawLine(x + 2, y - 2, x + 2, y + 2);
			gc.drawLine(x + 3, y - 1, x + 3, y + 1);
			gc.drawLine(x + 4, y, x + 4, y);
			this.mRightArrowBounds = new Rectangle(x - 4, y - 4, x + 8, y);
			break;
		case ARROW_LEFT:
			gc.drawLine(x - 1, y, x - 1, y);
			gc.drawLine(x, y - 1, x, y + 1);
			gc.drawLine(x + 1, y - 2, x + 1, y + 2);
			gc.drawLine(x + 2, y - 3, x + 2, y + 3);
			gc.drawLine(x + 3, y - 4, x + 3, y + 4);
			this.mLeftArrowBounds = new Rectangle(x - 4, y - 4, x + 8, y);
			break;
		}
	}

	private static boolean isInside(int x, int y, Rectangle rect) {
		if (rect == null) {
			return false;
		}

		if (x >= rect.x && y >= rect.y && x <= (rect.x + rect.width) && y <= (rect.y + rect.height)) {
			return true;
		}

		return false;
	}

	public void mouseMove(MouseEvent e) {

		// "dragging" the day, just paint it
		if (e.stateMask != 0) {
			doDaySelection(e.x, e.y);
		}

		if (this.mArrowRun && this.mArrowThread != null) {
			if (!isInside(e.x, e.y, this.mLeftArrowBounds) && !isInside(e.x, e.y, this.mRightArrowBounds)) {
				this.mArrowPause = true;
				// also pause the speed
				this.mArrowIterations = 0;
				this.mArrowSleepTime = ARROW_SLOW_TIME;
			} else {
				if (isInside(e.x, e.y, this.mLeftArrowBounds)) {
					this.mArrowThreadDirection = this.ARROW_LEFT;
				} else {
					this.mArrowThreadDirection = this.ARROW_RIGHT;
				}

				this.mArrowPause = false;
			}
		}
	}

	public void mouseDoubleClick(MouseEvent event) {
	}

	// draw the date selection on mouse down to give that flicker of "response"
	// to the user
	public void mouseDown(MouseEvent event) {

		if (isInside(event.x, event.y, this.mLeftArrowBounds)) {
			prevMonth();

			runArrowThread(this.ARROW_LEFT);

			return;
		}

		if (isInside(event.x, event.y, this.mRightArrowBounds)) {
			nextMonth();

			runArrowThread(this.ARROW_RIGHT);

			return;
		}

		doDaySelection(event.x, event.y);

	}

	private void killArrowThread() {
		if (this.mArrowThread != null) {
			this.mArrowPause = true;
			this.mArrowThreadDirection = 0;
			this.mArrowRun = false;
			this.mArrowThread = null;
		}
	}

	private void runArrowThread(int direction) {
		this.mArrowThreadDirection = direction;
		this.mArrowIterations = 0;
		this.mArrowSleepTime = ARROW_SLOW_TIME;
		this.mArrowRun = false;
		this.mArrowPause = false;
		this.mArrowThread = new Thread() {
			public void run() {
				while (CalendarComposite2.this.mArrowRun) {
					try {
						sleep(CalendarComposite2.this.mArrowSleepTime);

						if (!CalendarComposite2.this.mArrowPause) {
							CalendarComposite2.this.mArrowIterations++;
						}

						if (CalendarComposite2.this.mArrowIterations > ARROW_SPEED_SWITCH_COUNT && CalendarComposite2.this.mArrowSleepTime != ARROW_FAST_TIME) {
							CalendarComposite2.this.mArrowSleepTime = ARROW_FAST_TIME;
						}
					} catch (InterruptedException e) {
						e.printStackTrace();
					}
					if (!CalendarComposite2.this.mArrowPause) {
						Display.getDefault().syncExec(new Runnable() {
							public void run() {
								if (isDisposed()) {
									CalendarComposite2.this.mArrowRun = false;
									CalendarComposite2.this.mArrowThread = null;
								} else {
									if (CalendarComposite2.this.mArrowThreadDirection == CalendarComposite2.this.ARROW_LEFT) {
										prevMonth();
									} else if (CalendarComposite2.this.mArrowThreadDirection == CalendarComposite2.this.ARROW_RIGHT) {
										nextMonth();
									}
								}
							}
						});
					}
				}
			}
		};
		this.mArrowPause = false;
		this.mArrowRun = true;
		this.mArrowThread.start();
	}

	// selects a day at x, y if there is a day there to select
	private void doDaySelection(int x, int y) {
		int curYear = this.mDateFactory.getYear(this.mCalendar);
		int curMonth = this.mDateFactory.getMonth(this.mCalendar);
		this.mNoDayClicked = false;

		for (int i = 0; i < this.mDays.length; i++) {
			if (isInside(x, y, this.mDays[i].bounds)) {

				int year = this.mDateFactory.getYear(this.mDays[i].date);
				int month = this.mDateFactory.getMonth(this.mDays[i].date);

				if (year == curYear) {
					if (month < curMonth) {
						prevMonth();
						return;
					} else if (month > curMonth) {
						nextMonth();
						return;
					}
				} else {
					if (year < curYear) {
						prevMonth();
						return;
					} else if (year > curYear) {
						nextMonth();
						return;
					}
				}

				if (this.mSelectedDay == null) {
					this.mSelectedDay = this.mDateFactory.clone(this.mDays[i].date);
				} else {
					this.mSelectedDay = this.mDateFactory.clone(this.mDays[i].date, this.mSelectedDay);
				}

				redraw();
				return;
			}
		}

		this.mNoDayClicked = true;
	}

	public void mouseUp(MouseEvent event) {
		killArrowThread();

		if (this.mNoDayClicked) {
			this.mNoDayClicked = false;
			return;
		}

		if (this.mSelectedDay != null) {
			notifyListeners();
		}
	}

	private void notifyListeners() {
		final Object date = this.mSelectedDay == null ? null : this.mDateFactory.clone(this.mSelectedDay);
		for (int i = 0; i < this.mListeners.size(); i++) {
			IDateListener listener = (IDateListener) this.mListeners.get(i);
			listener.dateChanged(date);
		}
	}

	public void addDateListener(IDateListener listener) {
		if (!this.mListeners.contains(listener)) {
			this.mListeners.add(listener);
		}
	}

	public void removeCalendarListener(IDateListener listener) {
		this.mListeners.remove(listener);
	}

	private void keyPressed(int keyCode, int stateMask) {
		if (this.mSelectedDay == null) {
			this.mSelectedDay = this.mDateFactory.clone(this.mCalendar);
			this.mSelectedDay = this.mDateFactory.withDay(this.mSelectedDay, 1);
			redraw();
		} else {
			if (keyCode == SWT.ARROW_RIGHT) {
				int monthNow = this.mDateFactory.getMonth(this.mSelectedDay);
				this.mSelectedDay = this.mDateFactory.plusDay(this.mSelectedDay, 1);
				int monthAfter = this.mDateFactory.getMonth(this.mSelectedDay);
				if (monthAfter != monthNow) {
					this.mCalendar = this.mDateFactory.plusMonth(this.mCalendar, 1);
				}
				redraw();
			} else if (keyCode == SWT.ARROW_LEFT) {
				int monthNow = this.mDateFactory.getMonth(this.mSelectedDay);
				this.mSelectedDay = this.mDateFactory.plusDay(this.mSelectedDay, -1);
				int monthAfter = this.mDateFactory.getMonth(this.mSelectedDay);
				if (monthAfter != monthNow) {
					this.mCalendar = this.mDateFactory.plusMonth(this.mCalendar, -1);
				}
				redraw();
			} else if (keyCode == SWT.ARROW_UP) {
				int monthNow = this.mDateFactory.getMonth(this.mSelectedDay);
				this.mSelectedDay = this.mDateFactory.plusDay(this.mSelectedDay, -7);
				int monthAfter = this.mDateFactory.getMonth(this.mSelectedDay);
				if (monthAfter != monthNow) {
					this.mCalendar = this.mDateFactory.plusMonth(this.mCalendar, -1);
				}
				redraw();
			} else if (keyCode == SWT.ARROW_DOWN) {
				int monthNow = this.mDateFactory.getMonth(this.mSelectedDay);
				this.mSelectedDay = this.mDateFactory.plusDay(this.mSelectedDay, 7);
				int monthAfter = this.mDateFactory.getMonth(this.mSelectedDay);
				if (monthAfter != monthNow) {
					this.mCalendar = this.mDateFactory.plusMonth(this.mCalendar, 1);
				}
				redraw();
			} else if (keyCode == SWT.CR || keyCode == SWT.LF) {
				notifyListeners();
				return;
			}
		}
	}

	private class ButtonSectionLayout extends Layout {

		protected Point computeSize(Composite composite, int wHint, int hHint, boolean flushCache) {
			return new Point(153, 160);
		}

		protected void layout(Composite composite, boolean flushCache) {
			int bheight = CalendarComposite2.this.mSettings.getButtonHeight();
			int bwidth = CalendarComposite2.this.mSettings.getButtonWidth();
			int vspacer = CalendarComposite2.this.mSettings.getButtonVerticalSpace();
			int bspacer = CalendarComposite2.this.mSettings.getButtonsHorizontalSpace();
			if (CalendarCombo.OS_CARBON) {
				bwidth = CalendarComposite2.this.mSettings.getButtonWidthCarbon();
				bheight = CalendarComposite2.this.mSettings.getCarbonButtonHeight();
				vspacer = CalendarComposite2.this.mSettings.getCarbonButtonVerticalSpace();
				bspacer = CalendarComposite2.this.mSettings.getCarbonButtonsHorizontalSpace();
			}

			// see how much space we put on the left and right sides of the
			// buttons
			int width = CalendarComposite2.this.mSettings.getCalendarWidth() - (bwidth * 2) - bspacer;
			width /= 2;

			int button1Left = width;
			int button2Left = CalendarComposite2.this.mSettings.getCalendarWidth() - width - bwidth;

			Control[] children = composite.getChildren();
			for (int i = 0; i < children.length; i++) {
				switch (i) {
				case 0:
					children[i].setBounds(button1Left, vspacer, bwidth, bheight);

					break;
				case 1:
					children[i].setBounds(button2Left, vspacer, bwidth, bheight);
					break;
				}

			}
		}

	}

	private final class CalDay {
		public Object date;
		public int number;
		public Rectangle bounds;
		public CalDay(int number, Object date, Rectangle bounds) {
			this.date = date;
			this.bounds = bounds;
			this.number = number;
		}
	}

	public static void main(String[] args) {

		Display display = new Display();
		Shell shell = new Shell(display);
		shell.setLayout(new FillLayout());

		CalendarComposite2 calendar = new CalendarComposite2(shell, new MyColorManager(), new MySettings(), null);
		calendar.addDateListener(new IDateListener(){
			public void dateChanged(Object date) {
				System.out.println(date);
			}
		});

		shell.pack();
		shell.open();
		while (!shell.isDisposed()) {
			if (!display.readAndDispatch()) {
				display.sleep();
			}
		}
		display.dispose();

	}

	private static class MyColorManager extends AbstractColorManager {
		private MyColorManager() {
			super(SKIN_BLUE);
		}
	}

	private static class MySettings extends AbstractSettings {

		private Font macFont;

		MySettings() {
			super();
		}

		public Locale getLocale() {
			return Locale.getDefault();
		}

		public String getNoneText() {
			return "Aucun";
		}

		public String getTodayText() {
			return "Auj.";
		}

		public boolean showCalendarInRightCorner() {
			return true;
		}

		public Font getCarbonDrawFont() {
			if (CalendarCombo.OS_CARBON && (this.macFont == null || this.macFont.isDisposed())) {
				this.macFont = new Font(Display.getDefault(), "Lucida Grande", 11, SWT.NORMAL);
			}
			return this.macFont;
		}

	}
}
